package com.atlassian.seraph.config;

import com.atlassian.seraph.auth.RoleMapper;
import com.atlassian.seraph.auth.Authenticator;
import com.atlassian.seraph.auth.AuthenticationContext;
import com.atlassian.seraph.auth.AuthenticationContextImpl;
import com.atlassian.seraph.SecurityService;
import com.atlassian.seraph.Initable;
import com.atlassian.seraph.cookie.CookieFactory;
import com.atlassian.seraph.util.XMLUtils;
import com.atlassian.seraph.controller.SecurityController;
import com.atlassian.seraph.interceptor.Interceptor;
import com.opensymphony.util.ClassLoaderUtil;
import org.apache.log4j.Category;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.xml.parsers.DocumentBuilderFactory;
import java.io.Serializable;
import java.net.URL;
import java.util.*;

/**
 * The main class of Seraph's configuration.
 * <p>
 * This class is a Singleton, access it using SecurityConfigFactory.getInstance().
 */
public class SecurityConfigImpl implements Serializable, SecurityConfig
{
    private static final Category log = Category.getInstance(SecurityConfigImpl.class);
    private static SecurityConfigImpl instance = null;

    public static final String DEFAULT_CONFIG_LOCATION = "seraph-config.xml";

    private String configFileLocation = "seraph-config.xml";
    protected Authenticator authenticator = null;
    protected RoleMapper roleMapper = null;
    protected SecurityController controller;
    protected List services = null;
    protected List interceptors = null;

    private String loginURL;
    private String logoutURL;
    private String originalURLKey = "seraph_originalurl";
    private String cookieEncoding;
    private String loginCookieKey = "seraph.os.cookie";
    private String linkLoginURL;
    private String authType;
    private boolean insecureCookie;
    // the age of the autologin cookie - default = 1 year (in seconds)
    private int autoLoginCookieAge = 365 * 24 * 60 * 60;

    private LoginUrlStrategy loginUrlStrategy;
    private String loginCookiePath;


    public SecurityConfigImpl(String configFileLocation) throws ConfigurationException
    {
        if (configFileLocation != null)
        {
            this.configFileLocation = configFileLocation;
            log.debug("Config file location passed.  Location: " + this.configFileLocation);
        }
        else
        {
            log.debug("Initialising securityConfig using default configFile: " + this.configFileLocation);
        }

        init();
    }

    private void init() throws ConfigurationException
    {
        services = new ArrayList();
        interceptors = new ArrayList();

        try
        {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            URL fileUrl = ClassLoaderUtil.getResource(configFileLocation, this.getClass());

            if (fileUrl == null)
                throw new IllegalArgumentException("No such XML file: " + configFileLocation);

            // Parse document
            org.w3c.dom.Document doc = factory.newDocumentBuilder().parse(fileUrl.toString());
            Element rootEl = doc.getDocumentElement();

            configureParameters(rootEl);
            configureAuthenticator(rootEl);
            configureController(rootEl);
            configureRoleMapper(rootEl);
            configureServices(rootEl);
            configureInterceptors(rootEl);
            configureLoginUrlStrategy(rootEl);
            CookieFactory.init(this);
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new ConfigurationException("Exception configuring from '"+configFileLocation+"': " + e);
        }
    }

    protected void configureLoginUrlStrategy(Element rootEl) throws ConfigurationException
    {
        loginUrlStrategy = (LoginUrlStrategy) configureClass(rootEl, "login-url-strategy");

        if (loginUrlStrategy == null)
        {
            loginUrlStrategy = new DefaultLoginUrlStrategy();
        }
    }

    protected void configureAuthenticator(Element rootEl) throws ConfigurationException
    {
        authenticator = (Authenticator) configureClass(rootEl, "authenticator");

        try
        {
            if (authenticator == null)
            {
                authenticator = (Authenticator) ClassLoaderUtil.loadClass(Authenticator.DEFAULT_AUTHENTICATOR, this.getClass()).newInstance();
                authenticator.init(new HashMap(), this);
            }
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Could not lookup class: " + Authenticator.DEFAULT_AUTHENTICATOR + ":" + e);
        }
    }

    protected void configureController(Element rootEl) throws ConfigurationException
    {
        controller = (SecurityController) configureClass(rootEl, "controller");

        try
        {
            if (controller == null)
                controller = (SecurityController) ClassLoaderUtil.loadClass(SecurityController.NULL_CONTROLLER, this.getClass()).newInstance();
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Could not lookup class: " + SecurityController.NULL_CONTROLLER + ":" + e);
        }
    }

    protected void configureRoleMapper(Element rootEl) throws ConfigurationException
    {
        roleMapper = (RoleMapper) configureClass(rootEl, "rolemapper");
    }

    private Initable configureClass(Element rootEl, String tagname) throws ConfigurationException
    {
        try
        {
            NodeList elementList = rootEl.getElementsByTagName(tagname);

            for (int i = 0; i < elementList.getLength(); i++)
            {
                Element authEl = (Element) elementList.item(i);
                String clazz = authEl.getAttribute("class");

                Initable initable = (Initable) ClassLoaderUtil.loadClass(clazz, this.getClass()).newInstance();

                Map params = getInitParameters(authEl);

                initable.init(params, this);
                return initable;
            }
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Could not create: " + tagname + ": " + e);
        }

        return null;
    }

    private void configureParameters(Element rootEl)
    {
        NodeList nl = rootEl.getElementsByTagName("parameters");
        Element parametersEl = (Element) nl.item(0);
        Map globalParams = getInitParameters(parametersEl);

        loginURL = (String) globalParams.get("login.url");
        linkLoginURL = (String) globalParams.get("link.login.url");
        logoutURL = (String) globalParams.get("logout.url");

        if (globalParams.get("original.url.key") != null)
            originalURLKey = (String) globalParams.get("original.url.key");

        if (globalParams.get("cookie.encoding") != null)
            cookieEncoding = (String) globalParams.get("cookie.encoding");

        if (globalParams.get("login.cookie.key") != null)
            loginCookieKey = (String) globalParams.get("login.cookie.key");

        if (globalParams.get("login.cookie.path") != null)
            loginCookiePath = (String) globalParams.get("login.cookie.path");

        if (globalParams.get("authentication.type") != null)
            authType = (String) globalParams.get("authentication.type");

        if (globalParams.get("autologin.cookie.age") != null)
            autoLoginCookieAge = Integer.parseInt(globalParams.get("autologin.cookie.age").toString());

        if (globalParams.get("insecure.cookie") != null)
            insecureCookie = "true".equals(globalParams.get("insecure.cookie"));
    }

    private void configureServices(Element rootEl) throws ConfigurationException
    {
        NodeList nl = rootEl.getElementsByTagName("services");

        if (nl != null && nl.getLength() > 0)
        {
            Element servicesEl = (Element) nl.item(0);
            NodeList serviceList = servicesEl.getElementsByTagName("service");

            for (int i = 0; i < serviceList.getLength(); i++)
            {
                Element serviceEl = (Element) serviceList.item(i);
                String serviceClazz = serviceEl.getAttribute("class");

                if (serviceClazz == null || "".equals(serviceClazz))
                    throw new ConfigurationException("Service element with bad class attribute");

                try
                {
                    log.debug("Adding seraph service of class: " + serviceClazz);
                    SecurityService service = (SecurityService) ClassLoaderUtil.loadClass(serviceClazz, this.getClass()).newInstance();

                    Map serviceParams = getInitParameters(serviceEl);

                    service.init(serviceParams, this);

                    services.add(service);
                }
                catch (Exception e)
                {
                    e.printStackTrace();
                    throw new ConfigurationException("Could not getRequest service: " + serviceClazz + ". Exception: " + e);
                }
            }
        }
    }

    protected void configureInterceptors(Element rootEl) throws ConfigurationException
    {
        NodeList nl = rootEl.getElementsByTagName("interceptors");

        if (nl != null && nl.getLength() > 0)
        {
            Element interceptorsEl = (Element) nl.item(0);
            NodeList interceptorList = interceptorsEl.getElementsByTagName("interceptor");

            for (int i = 0; i < interceptorList.getLength(); i++)
            {
                Element interceptorEl = (Element) interceptorList.item(i);
                String interceptorClazz = interceptorEl.getAttribute("class");

                if (interceptorClazz == null || "".equals(interceptorClazz))

                    throw new ConfigurationException("Interceptor element with bad class attribute");

                try
                {
                    log.debug("Adding interceptor of class: " + interceptorClazz);
                    Interceptor interceptor = (Interceptor) ClassLoaderUtil.loadClass(interceptorClazz, this.getClass()).newInstance();

                    Map interceptorParams = getInitParameters(interceptorEl);

                    interceptor.init(interceptorParams, this);

                    interceptors.add(interceptor);
                }
                catch (Exception e)
                {
                    e.printStackTrace();
                    throw new ConfigurationException("Could not getRequest service: " + interceptorClazz + ". Exception: " + e);
                }
            }
        }
    }

    protected Map getInitParameters(Element el)
    {
        Map params = new HashMap();

        NodeList nl = el.getElementsByTagName("init-param");

        for (int i = 0; i < nl.getLength(); i++)
        {
            Node initParam = nl.item(i);
            String paramName = XMLUtils.getContainedText(initParam, "param-name");
            String paramValue = XMLUtils.getContainedText(initParam, "param-value");
            params.put(paramName, paramValue);
        }

        return params;
    }

    public void destroy()
    {
        for (Iterator iterator = services.iterator(); iterator.hasNext();)
        {
            SecurityService securityService = (SecurityService) iterator.next();
            securityService.destroy();
        }

        for (Iterator iterator = interceptors.iterator(); iterator.hasNext();)
        {
            ((Interceptor) iterator.next()).destroy();
        }

        return;
    }

    public void addInterceptor(Interceptor interceptor)
    {
        interceptors.add(interceptor);
    }

    public List getServices()
    {
        return services;
    }

    public String getLoginURL()
    {
        return loginUrlStrategy.getLoginURL(this, loginURL);
    }

    public String getLinkLoginURL()
    {
        return loginUrlStrategy.getLinkLoginURL(this, linkLoginURL);
    }

    public String getLogoutURL()
    {
        return loginUrlStrategy.getLogoutURL(this, logoutURL);
    }

    public String getOriginalURLKey()
    {
        return originalURLKey;
    }

    public Authenticator getAuthenticator()
    {
        return authenticator;
    }

    public AuthenticationContext getAuthenticationContext()
    {
        return new AuthenticationContextImpl(); //todo - load this from a config file
    }

    public SecurityController getController()
    {
        return controller;
    }

    /** Instantiates a SecurityConfigImpl. If you just want a {@link SecurityConfig}, use
     * {@link SecurityConfigFactory#getInstance(java.lang.String)} instead.
     */
    public static SecurityConfigImpl getInstance(String configFileLocation) throws ConfigurationException
    {
        instance = new SecurityConfigImpl(configFileLocation);
        return instance;
    }

    /** Instantiates a SecurityConfigImpl using the default config file.
     *  If you just want a {@link SecurityConfig}, use
     * {@link SecurityConfigFactory#getInstance()} instead.
     */
    public static SecurityConfig getInstance()
    {
        if (instance == null)
        {
            try
            {
                if (instance == null)
                {
                    instance = new SecurityConfigImpl(DEFAULT_CONFIG_LOCATION);
                }
            }
            catch (ConfigurationException e)
            {
                log.error("Could not configure SecurityConfigImpl instance: " + e, e);
            }
        }

        return instance;
    }

    public RoleMapper getRoleMapper()
    {
        return roleMapper;
    }

    public List getInterceptors(Class desiredInterceptorClass)
    {
        List result = new ArrayList();

        for (Iterator iterator = interceptors.iterator(); iterator.hasNext();)
        {
            Interceptor interceptor = (Interceptor) iterator.next();

            if (desiredInterceptorClass.isAssignableFrom(interceptor.getClass()))
            {
                result.add(interceptor);
            }
        }

        return result;
    }

    public String getCookieEncoding()
    {
        return cookieEncoding;
    }

    public String getLoginCookiePath()
    {
        return loginCookiePath;
    }

    public String getLoginCookieKey()
    {
        return loginCookieKey;
    }

    public String getAuthType()
    {
        return authType;
    }

    public boolean isInsecureCookie()
    {
        return insecureCookie;
    }

    public int getAutoLoginCookieAge()
    {
        return autoLoginCookieAge;
    }
}
