package com.atlassian.security.auth.trustedapps;

import com.atlassian.security.auth.trustedapps.Transcoder.Base64Transcoder;

import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public class ListApplicationRetriever implements ApplicationRetriever
{
    private final List /* <String> */values;
    private final EncryptionProvider encryptionProvider;
    private final Transcoder transcoder = new Base64Transcoder();

    ListApplicationRetriever(EncryptionProvider encryptionProvider, List /* <String> */values)
    {
        Null.not("encryptionProvider", encryptionProvider);
        Null.not("values", values);
        int i = 0;
        for (Iterator it = values.iterator(); it.hasNext();)
        {
            Null.not("value: " + i++, it.next());
        }

        this.encryptionProvider = encryptionProvider;
        this.values = new ArrayList /* <String> */(values);
    }

    public Application getApplication() throws RetrievalException
    {
        if (values.size() < 2)
        {
            throw new ApplicationNotFoundException("Application Certificate too small");
        }
        if (values.size() == 2)
        {
            return getApplicationProtocolV0();
        }
        return getApplicationProtocolV1();
    }

    private Application getApplicationProtocolV1() throws RetrievalException
    {
        // decorate the protocol zero version
        Application result = getApplicationProtocolV0();
        // with some validation of the certificate
        String protocol = (String) values.get(2);
        String magic = (String) values.get(3);
        try
        {
            final Integer protocolVersion = isBlank(protocol) ? null : Integer.valueOf(protocol);
            try
            {
                TrustedApplicationUtils.validateMagicNumber("application details", result.getID(), protocolVersion, magic);
            }
            catch (InvalidCertificateException e)
            {
                throw new InvalidApplicationDetailsException(e);
            }
        }
        catch (NumberFormatException e)
        {
            throw new InvalidApplicationDetailsException(e);
        }
        return result;
    }

    private Application getApplicationProtocolV0() throws RetrievalException
    {
        try
        {
            String id = (String) values.get(0);
            String keyStr = (String) values.get(1);

            if (keyStr == null)
            {
                throw new ApplicationNotFoundException("Public Key not found");
            }

            final byte[] data = transcoder.decode(keyStr);
            final PublicKey key = encryptionProvider.toPublicKey(data);
            return new SimpleApplication(id, key);
        }
        catch (InvalidKeySpecException e)
        {
            throw new RuntimeException(e);
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new RuntimeException(e);
        }
        catch (NoSuchProviderException e)
        {
            throw new RuntimeException(e);
        }
    }

    private static boolean isBlank(String input)
    {
        return (input == null) || input.trim().length() == 0;
    }
}