package com.atlassian.security.auth.trustedapps;

import java.security.PublicKey;

import javax.servlet.http.HttpServletRequest;

/**
 * very basic implementation
 */
public class DefaultTrustedApplication implements TrustedApplication
{
    protected final String id;
    protected final PublicKey publicKey;
    protected final long certificateTimeout;
    protected final RequestValidator requestValidator;
    protected final EncryptionProvider encryptionProvider;

    public DefaultTrustedApplication(EncryptionProvider encryptionProvider, PublicKey publicKey, String id, long certificateTimeout, RequestValidator requestValidator)
    {
        Null.not("encryptionProvider", encryptionProvider);
        Null.not("publicKey", publicKey);
        Null.not("id", id);
        Null.not("requestValidator", requestValidator);

        this.encryptionProvider = encryptionProvider;
        this.publicKey = publicKey;
        this.id = id;
        this.certificateTimeout = certificateTimeout;
        this.requestValidator = requestValidator;
    }

    public DefaultTrustedApplication(EncryptionProvider encryptionProvider, PublicKey key, String id, long certificateTimeout, URLMatcher urlMatcher, IPMatcher ipMatcher)
    {
        this(encryptionProvider , key, id, certificateTimeout, new DefaultRequestValidator(ipMatcher, urlMatcher));
    }

    public DefaultTrustedApplication(PublicKey key, String id, long certificateTimeout, URLMatcher urlMatcher, IPMatcher ipMatcher)
    {
        this(new BouncyCastleEncryptionProvider(), key, id, certificateTimeout, urlMatcher, ipMatcher);
    }

    public ApplicationCertificate decode(EncryptedCertificate encCert, HttpServletRequest request) throws InvalidCertificateException
    {
        ApplicationCertificate certificate = encryptionProvider.decodeEncryptedCertificate(encCert, publicKey, getID());

        checkCertificateExpiry(certificate);
        checkRequest(request);

        return certificate;
    }

    public String getID()
    {
        return id;
    }

    public PublicKey getPublicKey()
    {
        return publicKey;
    }

    private void checkCertificateExpiry(ApplicationCertificate certificate) throws InvalidCertificateException
    {
        if (certificate.getCreationTime().getTime() + certificateTimeout <= System.currentTimeMillis())
        {
            throw new CertificateTooOldException(certificate, certificateTimeout);
        }
    }
    
    private void checkRequest(HttpServletRequest request) throws InvalidCertificateException
    {
        try
        {
            requestValidator.validate(request);
        }
        catch (InvalidRequestException e)
        {
            throw new InvalidCertificateException(e);
        }
    }
}