package com.atlassian.sal.api.pluginsettings;

import com.atlassian.annotations.PublicApi;

/**
 * Defines a key-value store for a specific namespace, as constructed by {@link PluginSettingsFactory}.
 * <p>
 * This interface provides methods to store, retrieve, and remove settings within a namespace. Settings are stored as
 * key-value pairs where keys are strings and values can be of various supported types.
 * <p>
 * <b>Key Constraints:</b>
 * <ul>
 * <li>Keys cannot be {@code null}</li>
 * <li>Keys must not exceed {@value MAX_KEY_LENGTH} characters in length</li>
 * </ul>
 * <p>
 * <b>Supported Value Types:</b>
 * <ul>
 * <li>{@link Integer}</li>
 * <li>{@link Long}</li>
 * <li>{@link Boolean}</li>
 * <li>{@link String}</li>
 * <li>{@link java.util.List}{@code <String>}</li>
 * <li>{@link java.util.Set}{@code <String>}</li>
 * <li>{@link java.util.Map}{@code <String, String>}</li>
 * <li>{@link java.util.Properties}</li>
 * </ul>
 * <p>
 * All values are serialized for storage and must not exceed {@value MAX_SERIALIZED_VALUE_LENGTH}
 * characters when serialized.
 * <p>
 * Implementations make no guarantees regarding thread-safety.
 *
 * @since 2.0
 * @see PluginSettingsFactory
 */
public interface PluginSettings {

    @PublicApi
    int MAX_KEY_LENGTH = 255;

    @PublicApi
    int MAX_SERIALIZED_VALUE_LENGTH = 99_000;

    /**
     * Retrieves the value associated with the specified key.
     *
     * @param key the setting key; must not be {@code null} nor exceed {@value MAX_KEY_LENGTH} characters
     * @return the setting value, or {@code null} if no value is associated with the key
     * @throws IllegalArgumentException if the key is {@code null} or invalid
     */
    Object get(String key);

    /**
     * Stores a value for the specified key, replacing any existing value.
     * <p>
     * The value must be one of the supported types. All values are serialized for storage and must not exceed
     * {@value MAX_SERIALIZED_VALUE_LENGTH} characters when serialized.
     * <p>
     * Passing {@code null} as the value will remove the setting, equivalent to calling {@link #remove(String)}.
     *
     * @param key the setting key; must not be {@code null} nor exceed {@value MAX_KEY_LENGTH} characters
     * @param value the setting value; must be one of the supported types, or {@code null} to remove the setting.
     *              When serialized, the value must not exceed {@value MAX_SERIALIZED_VALUE_LENGTH} characters
     * @return the previous value associated with the key, or {@code null} if no value existed
     * @throws IllegalArgumentException if the key is {@code null} or invalid, or if the value is not a supported type
     *                                  or exceeds the maximum serialized length
     */
    Object put(String key, Object value);

    /**
     * Removes the setting associated with the specified key.
     *
     * @param key the setting key; must not be {@code null} nor exceed {@value MAX_KEY_LENGTH} characters
     * @return the value that was removed, or {@code null} if no value was associated with the key
     * @throws IllegalArgumentException if the key is {@code null} or invalid
     */
    Object remove(String key);
}
