package com.atlassian.renderer.wysiwyg.converter;

import com.atlassian.renderer.IconManager;
import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.WikiStyleRenderer;
import com.atlassian.renderer.util.NodeUtil;
import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.v2.components.TextConverter;
import com.atlassian.renderer.v2.components.phrase.ForceNewLineRendererComponent;
import com.atlassian.renderer.v2.macro.Macro;
import com.atlassian.renderer.v2.macro.MacroManager;
import com.atlassian.renderer.v2.macro.ResourceAwareMacroDecorator;
import com.atlassian.renderer.wysiwyg.ListContext;
import com.atlassian.renderer.wysiwyg.NodeContext;
import com.atlassian.renderer.wysiwyg.Styles;
import com.atlassian.renderer.wysiwyg.WysiwygConverter;
import com.atlassian.renderer.wysiwyg.WysiwygMacroHelper;
import com.atlassian.renderer.wysiwyg.WysiwygNodeConverter;
import com.opensymphony.util.TextUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.html.dom.HTMLDocumentImpl;
import org.apache.xerces.xni.parser.XMLDocumentFilter;
import org.cyberneko.html.filters.Writer;
import org.cyberneko.html.parsers.DOMFragmentParser;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.Node;
import org.w3c.dom.html.HTMLDocument;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.FactoryConfigurationError;
import java.io.IOException;
import java.io.StringReader;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class DefaultWysiwygConverter implements WysiwygConverter {
    public static final String TEXT_SEPARATOR = "TEXTSEP";
    /**
     * @deprecated use #TEXT_SEPARATOR
     */
    public static final String TEXT_SEPERATOR = TEXT_SEPARATOR;
    private static final String MATCH_NON_BREAKING_SPACES = "\u00A0+";

    /**
     * Class given to paragraphs whose purpose is to allow somewhere for the cursor to go after a macro.
     */
    public static final String CURSOR_PLACEMENT_CLASS = "atl_conf_pad";

    /**
     * A Cursor placement paragraph provides a place to put the cursor after the final macro or table
     * on a page, or before the first macro or table on a page.  Most cursor placement paragraphs are
     * stripped out as being redundant by
     * {@link #convertWikiMarkupToXHtml(RenderContext, String)}.
     */
    public static final String CURSOR_PLACEMENT_PARAGRAPH = "<p class=\"" + CURSOR_PLACEMENT_CLASS + "\"> </p>";

    private MacroManager macroManager;
    protected static boolean debug = false;
    private IconManager iconManager;
    protected WikiStyleRenderer renderer;
    /**
     * This contains a set of macro names which we don't add to the markup, because they are handled in other ways.
     */
    protected Set<String> macrosToIgnore = new HashSet<String>();

    /**
     * List of Text Converters that escape any textual input that might be otherwise subsequently interpreted as
     * wiki markup on the return trip.  This should only be accessed via the accessor,
     * {@link #getTextConverterComponents()} which may be overridden.
     */
    private final List<TextConverter> textConverterComponents;

    public DefaultWysiwygConverter() {
        macrosToIgnore.add("color");
        this.textConverterComponents = Collections.emptyList();
    }

    public DefaultWysiwygConverter(List<TextConverter> textConverterComponents) {
        macrosToIgnore.add("color");
        this.textConverterComponents = textConverterComponents;
    }

    public void setWikiStyleRenderer(WikiStyleRenderer renderer) {
        this.renderer = renderer;
    }

    public void setMacroManager(MacroManager macroManager) {
        this.macroManager = macroManager;
    }

    /**
     * Determine the appropriate separation string to place between the wiki markup for the
     * previous node and the current node based on their type and the current table/list/heading
     * (etc.) context.
     *
     * @param current     the "type" of the current node, see {@link TypeBasedSeparation}
     * @param nodeContext the current nodeContext.
     * @return the separation string to place between the nodes' wiki markup.  May be empty.
     */
    public String getSeparator(String current, NodeContext nodeContext) {
        String prevType;
        Node previous = nodeContext.getPreviousSibling();
        if (previous == null) {
            prevType = null;
        } else if (NodeUtil.isTextNode(previous)) {
            prevType = "text";
        } else if (WysiwygMacroHelper.getMacroName(previous) != null) {
            String attributeName = WysiwygMacroHelper.MACRO_HAS_TRAILING_NEWLINE_ATTRIBUTE;
            return NodeUtil.getBooleanAttributeValue(previous, attributeName, false) ? "\n" : "";
        } else if (WysiwygMacroHelper.isMacroBody(previous) || WysiwygMacroHelper.isMacroTag(previous)) {
            prevType = "text";
        } else if (isUserNewline(previous)) {
            prevType = "userNewline";
        } else if (isForcedNewline(previous)) {
            prevType = "forcedNewline";
        } else if (NodeUtil.isList(previous)) {
            prevType = "list";
        } else {
            prevType = previous.getNodeName().toLowerCase();
            if (FormatConverter.STYLE_NODE_TYPES.contains(prevType)) {
                prevType = "text";
            } else if (isHeading(prevType)) {
                prevType = "heading";
            } else if (isEmoticon(previous, prevType)) {
                prevType = "emoticon";
            }
        }
        String debugStr1 = "";
        String debugStr2 = "";
        if (debug) {
            debugStr1 = "[" + prevType + "-" + current;
            debugStr2 = nodeContext.isInTable() + "," + nodeContext.isInListItem() + "]";
        }
        Separation separation = TypeBasedSeparation.getSeparation(prevType, current);
        String sep;

        if (nodeContext.isInHeading()) {
            sep = separation.getSeparator();
            if (sep != null)
                sep = sep.replace("\n", "");
        } else if (nodeContext.isInTable()) {
            sep = separation.getTableSeparator();
        } else if (nodeContext.isInListItem()) {
            sep = separation.getListSeparator();
        } else {
            sep = separation.getSeparator();
        }
        if (sep == null) {
            return debugStr1 + debugStr2;
        }
        return debugStr1 + sep + debugStr2;
    }

    /**
     * @deprecated Since 5.0. Use {@link #getSeparator(String, NodeContext)} instead. This method is quite broken
     * in 6.0 since it does not respect {@link NodeContext#isInHeading()}.
     */
    public String getSep(Node previous, String current, boolean inTable, boolean inList) {
        NodeContext.Builder contextBuilder = new NodeContext.Builder(previous).inTable(inTable).inListItem(inList);
        return getSeparator(current, contextBuilder.build());
    }

    private static boolean isEmoticon(Node node, String nodeName) {
        return nodeName.equals("img") && NodeUtil.attributeContains(node, "src", "/images/icons/emoticons/");
    }

    public String convertChildren(NodeContext nodeContext) {
        StringBuffer wikiText = new StringBuffer();
        //Node previousSibling = null;
        if (nodeContext.getNode() != null && nodeContext.getNode().getChildNodes() != null) {
            NodeContext childContext = nodeContext.getFirstChildNodeContextPreservingPreviousSibling();
            while (childContext != null) {
                String converted = convertNode(childContext);
                if (StringUtils.isNotEmpty(converted)) {
                    wikiText.append(converted);
                    childContext = nodeContext.getNodeContextForNextChild(childContext);
                } else {
                    childContext = nodeContext.getNodeContextForNextChildPreservingPreviousSibling(childContext);
                }
            }
        }
        return wikiText.toString();
    }

    /**
     * Converts the children of the node in the given node context.
     * <p>
     * It cascades the styles and nulls out the previous sibling before calling
     * {@link #convertChildren(Node, Styles, ListContext, boolean, boolean, boolean, boolean, Node)}.
     *
     * @deprecated Since 5.0. Use {@link #convertChildren(NodeContext)} instead.
     */
    public String convertChildren(Node node, Styles styles, ListContext listContext, boolean inTable, boolean inListItem, boolean ignoreText, boolean escapeWikiMarkup, Node previousSibling) {
        NodeContext.Builder contextBuilder = new NodeContext.Builder(node);
        contextBuilder.previousSibling(previousSibling);
        // note that the creation of a new Styles object is not a straight copy of the existing one.  It retrieves
        // new styles out of the attributes of the node.
        contextBuilder.styles(new Styles(node, styles));
        contextBuilder.listContext(listContext);
        contextBuilder.inTable(inTable).inListItem(inListItem).ignoreText(ignoreText).escapeWikiMarkup(escapeWikiMarkup);
        return convertChildren(contextBuilder.build());
    }

    // todo: Put this somewhere better.
    private static List<? extends Converter> CONVERTERS = Collections.unmodifiableList(Arrays.asList(
            CommentConverter.INSTANCE, IgnoreNodeAndConvertChildText.INSTANCE,
            com.atlassian.renderer.wysiwyg.converter.TextConverter.INSTANCE, ExternallyDefinedConverter.INSTANCE,
            BreakConverter.INSTANCE, ParagraphConverter.INSTANCE, FormatConverter.INSTANCE, SpanConverter.INSTANCE,
            FontConverter.INSTANCE, ListConverter.INSTANCE, ListItemConverter.INSTANCE, TableConverter.INSTANCE,
            TableBodyConverter.INSTANCE, TableRowConverter.INSTANCE, TableCellConverter.TD, TableCellConverter.TH,
            DivConverter.INSTANCE, HeadingConverter.INSTANCE, ImageConverter.INSTANCE, LinkConverter.INSTANCE,
            PreformattingConverter.INSTANCE, HorizontalRuleConverter.INSTANCE, IgnoreNodeAndChildren.INSTANCE,
            BlockQuoteConverter.INSTANCE
    ));

    /**
     * Converts the node in the given node context to wiki markup.
     */
    public String convertNode(NodeContext nodeContext) {
        for (Converter converter : CONVERTERS) {
            if (converter.canConvert(nodeContext)) {
                String converted = converter.convertNode(nodeContext, this);
                return converted == null ? "" : converted;
            }
        }
        return getRawChildText(nodeContext.getNode(), true);
    }

    public static boolean isUserNewline(Node node) {
        return node != null && node.getNodeName() != null && node.getNodeName().toLowerCase().equals("p") && node.getAttributes() != null && node.getAttributes().getNamedItem("user") != null &&
                containsNoUserContent(node);
    }

    private static boolean containsNoUserContent(Node node) {
        String rawText = getRawChildTextWithoutReplacement(node);
        return StringUtils.isBlank(rawText) || rawText.trim().matches(MATCH_NON_BREAKING_SPACES);
    }

    WysiwygNodeConverter findNodeConverter(String converterName) {
        if (converterName.startsWith("macro:")) {
            String[] parts = converterName.split(":");
            if (parts.length != 2) {
                throw new RuntimeException("Illegal node converter name:'" + converterName + "'");
            }
            Macro m = macroManager.getEnabledMacro(parts[1]);
            if (m instanceof ResourceAwareMacroDecorator) {
                m = ((ResourceAwareMacroDecorator) m).getMacro();
            }
            if (!(m instanceof WysiwygNodeConverter)) {
                throw new RuntimeException("Macro '" + parts[1] + "' implemented by " + m.getClass() + " does not implement WysiwygNodeConverter.");
            }
            return (WysiwygNodeConverter) m;
        } else {
            throw new RuntimeException("Unrecognized node converter name:'" + converterName + "'");
        }
    }

    static boolean isHeading(String name) {
        return name.startsWith("h") && name.length() == 2 && Character.isDigit(name.charAt(1));
    }

    static boolean isForcedNewline(Node node) {
        return node != null && node.getNodeName() != null && node.getNodeName().toLowerCase().equals("br")
                && node.getAttributes() != null
                && ForceNewLineRendererComponent.FORCED_NEWLINE_CLASS.equals(NodeUtil.getAttribute(node, "class"));
    }


    /**
     * Return the text content of a node, adding newlines for &lt;br&gt; and &lt;p&gt; tags
     *
     * @param node          the node to get the text content from
     * @param stripNewlines if this is true then strip any newlines from the raw text, but still add newlines for br and p.
     * @return the converted text as a string.  This may be empty but will not be null.
     */
    public static String getRawChildText(Node node, boolean stripNewlines) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < node.getChildNodes().getLength(); ++i) {
            Node n = node.getChildNodes().item(i);
            if (NodeUtil.isTextNode(n)) {
                String s = n.getNodeValue();
                if (stripNewlines) {
                    // RNDR-82 Remove text nodes that are just a newline created by internet explorer for
                    // prettiness, RNDR-84 and other, arguably superfluous whitespace, while converting mid-paragraph
                    // newlines to standard spaces.  When newlines are important they should be accompanied by br 
                    // tags anyway, see below.
                    s = s.replaceAll("(\n|\r)", " ").trim();
                }
                sb.append(s);
            } else if (getNodeName(n).equals("br")) {
                sb.append("\n");
            }
            sb.append(getRawChildText(n, stripNewlines));
            if (getNodeName(n).equals("p")) {
                sb.append("\n");
            }
        }
        return sb.toString();
    }

    /**
     * Return the text content of a node,  <strong>WITHOUT<strong/> adding newlines for &lt;br&gt; and &lt;p&gt; tags
     *
     * @param node the node to get the text content from
     */
    public static String getRawChildTextWithoutReplacement(Node node) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < node.getChildNodes().getLength(); ++i) {
            Node n = node.getChildNodes().item(i);
            if (NodeUtil.isTextNode(n)) {
                String s = n.getNodeValue();
                sb.append(s);
            }
            sb.append(getRawChildTextWithoutReplacement(n));
        }
        return sb.toString();
    }

    private static String getNodeName(Node node) {
        return node.getNodeName().toLowerCase();
    }


    public void setIconManager(IconManager iconManager) {
        this.iconManager = iconManager;
    }

    /**
     * @deprecated This method seems unused in renderer and will probably be removed in a future version.
     */
    public String getMacroInfoHtml(RenderContext context, String name, int xOffset, int yOffset) {
        return "<img alt=\"" + name + "\" style=\"float:left;margin-right:-32;opacity:0.75;position:relative;left:" + xOffset + "px;top:" + yOffset + "px;\" src=\"" + context.getSiteRoot() + "/includes/js/editor/plugins/confluence/info.png\"/>";
    }

    public String convertXHtmlToWikiMarkup(String xhtml) {
        if (!TextUtils.stringSet(xhtml)) {
            return "";
        }
        try {
            xhtml = RenderUtils.stripCarriageReturns(xhtml);

            DOMFragmentParser parser = new DOMFragmentParser();

            HTMLDocument document = new HTMLDocumentImpl();
            DocumentFragment fragment = document.createDocumentFragment();
            // NekoHTML doesn't seem to properly handle processing instructions created by MS Word pasted into IE -- a ProcessingInstructionImpl
            // node is created, but following HTML is ignored. So we strip the processing instruction first:
            xhtml = xhtml.replaceAll("<\\?xml.*?/>", "");
            InputSource inputSource = new InputSource(new StringReader(xhtml));
            try {
                parser.setFeature("http://cyberneko.org/html/features/balance-tags/document-fragment", true);
                if (debug) {
                    parser.setProperty("http://cyberneko.org/html/properties/filters", new XMLDocumentFilter[]{new Writer()});
                }
                parser.parse(inputSource, fragment);
            } catch (SAXException e) {
                throw new RuntimeException(e);
            }
            StringBuffer wikiText = new StringBuffer();

            NodeContext nodeContext = new NodeContext.Builder(fragment).build();
            wikiText.append(convertNode(nodeContext));
            // fix consecutive newlines and other white space
            // Strip out trailing whitespace before newlines, "     \n" (Modified to fix CONF-4490)
            if (debug) {
                return wikiText.toString();
            } else {
                // todo: Justify these replacements and preferably handle them at a
                // lower level.
                String s = wikiText.toString().replaceAll("[\\s&&[^\n]]*\n", "\n").trim();
                s = s.replaceAll(" (" + TEXT_SEPARATOR + ")+", " ");
                s = s.replaceAll("\n(" + TEXT_SEPARATOR + ")+", "\n");
                s = s.replaceAll("^(" + TEXT_SEPARATOR + ")+", "");
                s = s.replaceAll("\\[(" + TEXT_SEPARATOR + ")+", "[");
                s = s.replaceAll("(" + TEXT_SEPARATOR + ")+ ", " ");
                s = s.replaceAll("(" + TEXT_SEPARATOR + ")+", " ");
                s = s.replaceAll(" \n", "\n");
                // fix phrase style markers, making them simple ('*') instead of complex ('{*}') where possible.
                s = s.replaceAll(VALID_START + PHRASE_CLEANUP_REGEX, "$1");
                // This should be handled at a lower level.  Currently it causes a bug
                // where {*} in a noformat macro becomes * CONF-14842
                s = s.replaceAll(PHRASE_CLEANUP_REGEX + VALID_END, "$1");
                // remove nbsps at the end of a line (can lead to StackOverflow, disabled for now)
//                s = s.replaceAll("((\\s)?&nbsp;(?:[\\s&&[^\\n]])?)+(\\n|$)", "$2$3");
                return s;
            }
        } catch (FactoryConfigurationError e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static final String VALID_START = "(?<![}\\p{L}\\p{Nd}\\\\])";
    public static final String VALID_END = "(?![{\\p{L}\\p{Nd}])";

    private static final String PHRASE_CLEANUP_REGEX = "\\{((?:\\?\\?)|(?:\\*)|(?:\\^)|(?:~)|(?:_)|(?:-)|(?:\\+)|(?:\\{\\{)|(?:\\}\\}))\\}";

    public String convertWikiMarkupToXHtml(RenderContext ctx, String wikiMarkup) {
        ctx.setRenderingForWysiwyg(true);
        wikiMarkup = RenderUtils.stripCarriageReturns(wikiMarkup);
        String s = renderer.convertWikiToXHtml(ctx, wikiMarkup);
        // now we remove any *unnecessary* padding paragraphs
        s = s.replaceAll(CURSOR_PLACEMENT_PARAGRAPH + "\\s*<p", "<p");
        return s;
    }

    /**
     * @deprecated As of release 4.1, use {@link NodeUtil#getAttribute(org.w3c.dom.Node, String)}.
     */
    public String getAttribute(Node node, String name) {
        return NodeUtil.getAttribute(node, name);
    }

    /**
     * Can be overridden to provide a different list of content escapers.
     *
     * @return the current list of text converter components.
     */
    protected List<TextConverter> getTextConverterComponents() {
        return textConverterComponents;
    }

    // todo: replace with setter injection on converters
    MacroManager getMacroManager() {
        return macroManager;
    }

    // todo: replace with setter injection on converters
    IconManager getIconManager() {
        return iconManager;
    }
}
