package com.atlassian.renderer.wysiwyg;

import com.atlassian.renderer.links.Link;
import com.atlassian.renderer.links.UnpermittedLink;
import com.atlassian.renderer.links.UnresolvedLink;
import com.atlassian.renderer.util.RendererUtil;
import com.atlassian.renderer.wysiwyg.converter.DefaultWysiwygConverter;
import com.opensymphony.util.TextUtils;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Node;

import java.util.ArrayList;
import java.util.List;

/**
 * The WysiwygLinkHelper helps transform wiki links into html links and back again.  In order to perform this, it puts
 * additional information into the html link tags, beyond what it requires just to display the link.
 * <p>
 * The following is the list of additional attributes:
 * <li>linktype - always "raw".  This is a magical literal to define that it is a link understood by this helper. </li>
 * <li>wikidestination - the destination part of the link in wiki markup</li>
 * <li>originalalias - the wiki markup of the "alias"</li>
 * <li>aliasspecified - whether the alias was explicitly included in the wiki markup</li>
 * <li>wikititle - the tooltip part of the link in wiki markup</li>
 */
public class WysiwygLinkHelper {
    static final String LINK_TYPE_ATTR = "linktype";
    static final String RAW_LINK_TYPE_ATTR_VALUE = "raw";

    static final String WIKI_DESTINATION_ATTR = "wikidestination";
    static final String ORIGINAL_ALIAS_ATTR = "originalalias";
    static final String ALIAS_SPECIFIED_ATTR = "aliasspecified";
    static final String WIKI_TITLE_ATTR = "wikititle";

    static final String CLASS_ATTR = "class";
    public static final String ERROR_CLASS_ATTR_VALUE = "linkerror";

    public static String getLinkInfoAttributes(Link link) {
        // Encode the HTML here to fix characters that may interfere with the link text, e.g. " (CONF-4544)
        StringBuffer buffer = new StringBuffer();
        RendererUtil.appendAttribute(LINK_TYPE_ATTR, RAW_LINK_TYPE_ATTR_VALUE, buffer);
        RendererUtil.appendAttribute(WIKI_DESTINATION_ATTR, TextUtils.htmlEncode(link.getWikiDestination()), buffer);

        if (link.isAliasSpecified())
            RendererUtil.appendAttribute(ALIAS_SPECIFIED_ATTR, "true", buffer);
        else
            RendererUtil.appendAttribute(ORIGINAL_ALIAS_ATTR, TextUtils.htmlEncode(link.getLinkBody()), buffer);

        if (link.getWikiTitle() != null)
            RendererUtil.appendAttribute(WIKI_TITLE_ATTR, TextUtils.htmlEncode(link.getWikiTitle()), buffer);

        if (link instanceof UnpermittedLink || link instanceof UnresolvedLink)
            RendererUtil.appendAttribute(CLASS_ATTR, ERROR_CLASS_ATTR_VALUE, buffer);

        return buffer.toString();
    }

    public static String createLinkWikiText(Node node, String newAlias) {
        String linkType = getNodeAttributeValue(node, LINK_TYPE_ATTR);
        if (!linkType.equals(RAW_LINK_TYPE_ATTR_VALUE)) // not an Atlassian link
            return "";

        // strip off any text sperator -- it isn't needed at the beginning of a link alias
        if (newAlias.startsWith(DefaultWysiwygConverter.TEXT_SEPARATOR)) {
            newAlias = newAlias.substring(DefaultWysiwygConverter.TEXT_SEPARATOR.length());
        }
        newAlias = newAlias.trim();

        String wikiDestination = getNodeAttributeValue(node, WIKI_DESTINATION_ATTR);
        String originalAlias = getNodeAttributeValue(node, ORIGINAL_ALIAS_ATTR);
        boolean aliasSpecified = Boolean.parseBoolean(getNodeAttributeValue(node, ALIAS_SPECIFIED_ATTR));
        String wikititle = getNodeAttributeValue(node, WIKI_TITLE_ATTR);

        List<String> components = new ArrayList<String>(3);

        // alias
        if (aliasSpecified || !newAlias.equals(originalAlias)) {
            components.add(newAlias);
        }

        // destination
        components.add(wikiDestination);

        // title
        if (wikititle != null)
            components.add(wikititle);

        return "[" + StringUtils.join(components.iterator(), '|') + "]";
    }

    private static String getNodeAttributeValue(Node node, String attributeName) {
        Node wikititleNode = node.getAttributes().getNamedItem(attributeName);
        return (wikititleNode == null) ? null : wikititleNode.getNodeValue();
    }

}
