package com.atlassian.renderer.v2.macro.basic;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.v2.macro.BaseMacro;
import com.atlassian.renderer.v2.macro.MacroException;
import com.atlassian.renderer.wysiwyg.MacroBodyConverter;
import com.atlassian.renderer.wysiwyg.NodeContext;
import com.atlassian.renderer.wysiwyg.WysiwygConverter;
import com.opensymphony.util.TextUtils;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.util.Map;

/**
 * A macro to include verbatim HTML in a page. <b>DO NOT</b> enable this macro on sites where you do not
 * trust your users absolutely, as it opens you up to a world of pain (and cross-site scripting security
 * issues)
 */
public class InlineHtmlMacro extends BaseMacro implements MacroBodyConverter {
    public boolean isInline() {
        return false;
    }

    public boolean hasBody() {
        return true;
    }

    public RenderMode getBodyRenderMode() {
        return RenderMode.NO_RENDER;
    }

    public String execute(Map parameters, String body, RenderContext renderContext) throws MacroException {
        return body;
    }

    public String convertXhtmlToWikiMarkup(NodeContext nodeContext, WysiwygConverter wysiwygConverter) {
        String markup = writeNodeListHTML(nodeContext.getNode().getChildNodes());
        // The WysiwygMacroHelper handles the newlines around the content.
        markup = RenderUtils.trimInitialNewline(markup);
        markup = StringUtils.chomp(markup, "\n");

        return markup;
    }

    private String writeNodeListHTML(NodeList childNodes) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < childNodes.getLength(); ++i) {
            sb.append(writeNodeHTML(childNodes.item(i)));
        }
        return sb.toString();
    }

    private String writeNodeHTML(Node node) {
        if (node.getNodeType() == Node.TEXT_NODE) {
            return node.getNodeValue();
        } else {
            StringBuffer sb = new StringBuffer();
            return sb.append("<").append(node.getNodeName()).append(writeNodeAttributes(node)).append(">")
                    .append(writeNodeListHTML(node.getChildNodes()))
                    .append("</").append(node.getNodeName()).append(">").toString();
        }
    }

    private String writeNodeAttributes(Node node) {
        StringBuffer sb = new StringBuffer("");
        NamedNodeMap attrs = node.getAttributes();
        // getAttributes() can return null for nodes which are not elements
        if (attrs != null) {
            for (int i = 0; i < attrs.getLength(); ++i) {
                Node attr = attrs.item(i);
                if (TextUtils.stringSet(attr.getNodeValue())) {
                    sb.append(" ").append(attr.getNodeName()).append("=\"").append(attr.getNodeValue()).append("\"");
                } else {
                    sb.append(" ").append(attr.getNodeName());
                }
            }
        }
        return sb.toString();
    }

    /**
     * This class used to suppress surrounding tags, but now uses them.
     */
    public boolean suppressSurroundingTagDuringWysiwygRendering() {
        return false;
    }

    /**
     * Render the HTML.  (AKA, display body unaltered)
     */
    public boolean suppressMacroRenderingDuringWysiwyg() {
        return false;
    }
}
