package com.atlassian.renderer.v2.macro;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.TokenType;
import com.atlassian.renderer.v2.RenderMode;

import java.util.Map;

/**
 * A pluggable, programmatic module for inserting interesting things into a wiki page.
 *
 * @see <a href="http://confluence.atlassian.com/display/DOC/Macro+Plugins">Macro Plugins Documentation</a>
 */
public interface Macro {
    static final String RAW_PARAMS_KEY = ": = | RAW | = :";

    /**
     * Declares what sort of html elements are returned by the macro.
     * {@link TokenType#INLINE} elements will have paragraphs drawn around them, or be incorporated into paragraphs
     * they are included in.
     * {@link TokenType#BLOCK} elements will be treated as their own paragraph, splitting any paragraphs they
     * would otherwise be in.
     * {@link TokenType#INLINE_BLOCK} elements will not have paragraphs drawn around them, but they won't split any
     * paragraphs they appear in either.
     * <p>
     * <p>Macros that contain elements that shouldn't appear in a paragraph, such as pre, should return
     * {@link TokenType#BLOCK}. If it produces a span, replacement text or text decoration, return
     * {@link TokenType#INLINE}. If it affects logic of whether its children are displayed but doesn't add its own
     * content, or you aren't sure what to return, return {@link TokenType#INLINE_BLOCK}.
     *
     * @param parameters the parameters passed to the macro.  These are useful here for macros that take parameters
     *                   which dramatically alter the appearance of the macro.
     * @param body       the macro body is provided in case its content could have a bearing on the type of element
     *                   returned by the macro
     * @param context    the context is provided in case its content could have a bearing on the type of element
     *                   returned by the macro
     * @return the TokenType that matches the content produced by this macro
     */
    TokenType getTokenType(Map<String, Object> parameters, String body, RenderContext context);

    /**
     * Determine if the macro is an "inline" element in the resulting HTML. Inline elements will have paragraphs
     * drawn around them, or be incorporated into paragraphs they are included in. As a rule of thumb, if your
     * macro produces a paragraph, table or div (or so on), you should return false. If it produces a span,
     * replacement text or text decoration, return true.
     * <p>
     * <p>Most macros will want to return false here.
     *
     * @return true if the macro output is included inline within the surrounding HTML, false if it forms its
     * own HTML block.
     * @deprecated since Renderer 6 (Confluence 3.1) use {@link #getTokenType} instead, which will by
     * default call this method. However, for macros to be backwardly compatible they will still need to implement
     * {@link #isInline} too.
     */
    boolean isInline();

    /**
     * Determine if the macro is a one-shot macro, or one that takes a body. If this method returns false, the
     * renderer will NOT look for an end-tag for the macro. If the method returns true, the renderer will look
     * for an end-tag, but if the end-tag is not found then the macro will be processed with an empty body.
     */
    boolean hasBody();

    /**
     * If the macro has a body, return the mode in which the body of the macro
     * should be rendered. The body of the macro will be rendered <i>before</i>
     * the macro is executed!
     * <p>
     * <p>If this method returns null, it causes the macro processor to treat the macro
     * as one that returns wiki-text rather than HTML. The body of the macro will be
     * passed in un-rendered, and the macro's output will be inserted back into the
     * page for further normal processing by the wiki-engine.
     *
     * @return the RenderMode in which the body of this macro should be rendered, or null
     * if the macro is substituting wiki-text
     */
    RenderMode getBodyRenderMode();

    /**
     * Execute the macro. Macros should write any output to the writer (it will
     * be rendered in the RenderMode returned in {@link #getBodyRenderMode()}).
     * <p>
     * <p>Macros are expected to output HTML. The output of macros will not be subjected to any
     * further processing by the wiki-engine. If your macro produces wiki-text, you are responsible
     * for rendering that text to HTML yourself using a {@link com.atlassian.renderer.v2.SubRenderer}
     * or {@link com.atlassian.renderer.WikiStyleRenderer}. If your macro returns pure wiki-text, you
     * can force further processing in the normal chain by returning null from {@link #getBodyRenderMode}
     *
     * @param parameters    the parameters included in the macro
     * @param body          the content of the body of the macro
     * @param renderContext the rendering context in which the macro was executed
     * @return the output of the macro
     * @throws MacroException if the macro fails in some unremarkable way. If the
     *                        macro fails in a way that is important to the server maintainer
     *                        (i.e. something is badly wrong), throw a RuntimeException instead.
     */
    String execute(Map<String, Object> parameters, String body, RenderContext renderContext) throws MacroException;

    /**
     * Suppress surrounding div/span during Wysiwyg rendering. This should return true if the macro provides it's own
     * Wysiwyg processing HTML.
     */
    boolean suppressSurroundingTagDuringWysiwygRendering();

    /**
     * Suppress the rendering of the macro -- the macro's body may still be rendered (depending on the render mode of
     * the macro), but the HTML the macro adds will not be created.
     *
     * @return
     */
    boolean suppressMacroRenderingDuringWysiwyg();

    /**
     * Defines how the Macro's body should be rendered for Wysiwyg editors.  This method is only relevant for macros
     * that return {@code false} from {@link #suppressSurroundingTagDuringWysiwygRendering()},
     * {@code true} from {@link #suppressMacroRenderingDuringWysiwyg()} and {@code true} from {@link #hasBody()}.
     */
    WysiwygBodyType getWysiwygBodyType();
}
