package com.atlassian.renderer.v2.macro;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.TokenType;
import com.atlassian.renderer.v2.RenderMode;

import java.util.Map;

/**
 * Abstract implementation of {@link Macro} which should be extended by all plugin macros.  New methods added to
 * the {@link Macro} interface have default implementation added to this class, so {@link Macro}s that extend
 * {@link BaseMacro} are to some extend future-proofed to API changes.
 */
public abstract class BaseMacro implements Macro {
    /**
     * {@inheritDoc}
     * <p>
     * <p>The default implementation tries to maintain backward compatibility with most macros.  It returns
     * {@link TokenType#INLINE} for macros that return {@code true} from {@link #isInline()} and
     * {@link TokenType#INLINE_BLOCK} for macros that return {@code false}.
     *
     * @param parameters ignored
     * @param body       ignored
     * @param context    ignored
     */
    public TokenType getTokenType(Map parameters, String body, RenderContext context) {
        return isInline() ? TokenType.INLINE : TokenType.INLINE_BLOCK;
    }

    /**
     * {@inheritDoc}
     * <p>
     * <p>The default implementation now returns false, which has been a defacto default previously.
     *
     * @return false
     * @deprecated override {@link #getTokenType} to indicate inline or strict block behaviour. However, for
     * macros to be backwardly compatible they will still need to implement {@link #isInline} too.
     */
    public boolean isInline() {
        return false;
    }

    public boolean suppressSurroundingTagDuringWysiwygRendering() {
        return false;
    }

    public boolean suppressMacroRenderingDuringWysiwyg() {
        return true;
    }

    /**
     * {@inheritDoc}
     * <p>
     * <p>The default implementation returns {@link WysiwygBodyType#WIKI_MARKUP}
     * for most macros, but {@link WysiwygBodyType#PREFORMAT} for macros that
     * have a {@link #getBodyRenderMode()} of {@link RenderMode#NO_RENDER}.
     *
     * @return {@link WysiwygBodyType#WIKI_MARKUP} or {@link WysiwygBodyType#PREFORMAT}
     */
    public WysiwygBodyType getWysiwygBodyType() {
        RenderMode bodyMode = getBodyRenderMode();
        return (RenderMode.NO_RENDER.equals(bodyMode))
                ? WysiwygBodyType.PREFORMAT : WysiwygBodyType.WIKI_MARKUP;
    }
}
