package com.atlassian.renderer.v2;

import com.atlassian.renderer.v2.components.MacroTag;
import com.atlassian.renderer.v2.components.WikiContentHandler;
import com.atlassian.renderer.v2.macro.Macro;
import com.atlassian.renderer.v2.macro.MacroManager;

import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * A wiki parser which delegates the processing of certain markup to the
 * {@link com.atlassian.renderer.v2.components.WikiContentHandler}. Currently
 * the parser differentiates between macro body text and other text.
 */
public class WikiMarkupParser {
    WikiContentHandler wikiContentHandler;
    private MacroManager macroManager;

    public WikiMarkupParser(MacroManager macroManager, WikiContentHandler wikiContentHandler) {
        this.wikiContentHandler = wikiContentHandler;
        this.macroManager = macroManager;
    }

    public String parse(String wiki) {
        return parse(wiki, true);
    }

    public String parse(final String wiki, final boolean considerVariableFormat) {
        StringBuffer out = new StringBuffer(wiki.length());
        if (wiki.indexOf("{") == -1) {
            wikiContentHandler.handleText(out, wiki);
            return out.toString();
        }

        int lastStart = 0;
        boolean inEscape = false;

        for (int i = 0; i < wiki.length(); i++) {
            char c = wiki.charAt(i);
            if (!inEscape) {
                switch (c) {
                    case '\\':
                        inEscape = true;
                        continue;
                    case '{':
                        // ignore phrase formatting characters
                        if (wiki.length() > i + 1 && "{*?^_-+~".indexOf(wiki.charAt(i + 1)) != -1) {
                            i++;
                            continue;
                        } else if (considerVariableFormat && isPrecededBy(wiki, i, '$')) {
                            // JSDS-1122: ignore format of ${...} that would be considered as a variable substitution/replacement format
                            i++;
                            continue;
                        }

                        // Copy everything before the macro into the output buffer
                        wikiContentHandler.handleText(out, wiki.substring(lastStart, i));
                        lastStart = i + 1;

                        // Handle the macro, advancing the pointer to the end of the region we've handled.
                        // If no macro was found, we'll stay where we started, no harm done.
                        i = handlePotentialMacro(wiki, i, out, considerVariableFormat);
                        lastStart = i + 1;
                }
            } else {
                inEscape = false;
            }
        }

        if (lastStart < wiki.length()) {
            wikiContentHandler.handleText(out, wiki.substring(lastStart));
        }

        return out.toString();

    }

    private int handlePotentialMacro(final String wiki, int i, final StringBuffer out, final boolean considerVariableFormat) {
        MacroTag startTag = MacroTag.makeMacroTag(wiki, i);
        if (startTag != null) {
            Macro macro = getMacroByName(startTag.command);

            if (macro == null || macro.hasBody()) {
                setEndTagIfPresent(wiki, considerVariableFormat, startTag);
            }

            if (startTag.getEndTag() != null) {
                MacroTag endTag = startTag.getEndTag();
                String body = wiki.substring(startTag.endIndex + 1, endTag.startIndex);
                // if the body is only a newline, we don't need to double up on the newline after/before 
                if ("\n".equals(body) && startTag.isNewlineAfter() && endTag.isNewlineBefore()) {
                    endTag.removeNewlineBefore();
                }
                makeMacro(out, startTag, body);
                i = endTag.endIndex;
            } else {
                makeMacro(out, startTag, "");
                i = startTag.endIndex;
            }

        } else {
            out.append('{');
        }

        return i;
    }

    private void makeMacro(StringBuffer buffer, MacroTag startTag, String body) {
        wikiContentHandler.handleMacro(buffer, startTag, body);
    }

    private Macro getMacroByName(String name) {
        if (name == null) {
            return null;
        }

        return macroManager.getEnabledMacro(name.toLowerCase());
    }

    private void setEndTagIfPresent(final String wiki, final boolean considerVariableFormat, final MacroTag startTag) {
        boolean inEscape = false;

        for (int i = startTag.startIndex + startTag.originalText.length(); i < wiki.length(); i++) {
            char c = wiki.charAt(i);

            if (inEscape) {
                inEscape = false;
                continue;
            }

            if (c == '{') {
                if (considerVariableFormat && isPrecededBy(wiki, i, '$')) {
                    // JSDS-1122: ignore format of ${...} that would be considered as a variable substitution/replacement format
                    continue;
                }
                final MacroTag endTag = MacroTag.makeMacroTag(wiki, i);
                if (endTag != null && startTag.command.equals(endTag.command) && endTag.argString.length() == 0) {
                    startTag.setEndTag(endTag);
                    return;
                }
            } else if (c == '\\') {
                inEscape = true;
            }
        }
    }

    private static boolean isPrecededBy(final String wiki, final int index, char maybePreceded) {
        return isNotBlank(wiki) && index > 0 && wiki.charAt(index - 1) == maybePreceded;
    }

}
