package com.atlassian.renderer.v2;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.TokenType;

/**
 * Decorator for V2Renderer. It escapes all tag used internally in V2Renderer to prevent tag injection attacks
 *
 */
class TokenEscapingV2Renderer implements Renderer {
    private final Renderer renderer;
    //randomized sequence of char unlikely to show in input string which not occurs during processing by internal algoritm of V2Renderer
    private final String openingToken = "escASfuenIiFdfujy";
    private final String closingToken = "RftE";
    private boolean escapeInternalTokens;

    public TokenEscapingV2Renderer(Renderer renderer) {
        this(renderer, true);
    }

    public TokenEscapingV2Renderer(Renderer renderer, boolean escapeInternalTokens) {
        this.renderer = renderer;
        this.escapeInternalTokens = escapeInternalTokens;
    }

    @Override
    public String render(String originalContent, RenderContext renderContext) {
        return unescapeTokens(renderer.render(escapeTokens(originalContent), renderContext));
    }

    private String escapeTokens(String originalContent) {
        if(!escapeInternalTokens) {
            return originalContent;
        }
        String wiki = originalContent;
        for (TokenType tokenType : TokenType.values()) {
            wiki = wiki.replace(tokenType.getTokenMarker(), wrapOrdinalWithRandomness(tokenType));
        }
        return wiki;
    }

    private String wrapOrdinalWithRandomness(TokenType tokenType) {
        return openingToken + tokenType.ordinal() + closingToken;
    }

    private String unescapeTokens(String result) {
        String wiki = result;
        for (TokenType tokenType : TokenType.values()) {
            wiki = wiki.replace(wrapOrdinalWithRandomness(tokenType), tokenType.getTokenMarker());
        }
        return wiki;
    }

    @Override
    public String renderAsText(String originalContent, RenderContext context) {
        return unescapeTokens(renderer.renderAsText(escapeTokens(originalContent), context));
    }

    @Override
    public String getRendererType() {
        return renderer.getRendererType();
    }

    public void setEscapeInternalTokens(boolean escapeInternalTokens) {
        this.escapeInternalTokens = escapeInternalTokens;
    }
}
