package com.atlassian.renderer.util;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.function.Supplier;

public class DefaultUnicodeEscapingTool implements UnicodeEscapingTool {
    private static final Set<Integer> DEFAULT_BIDI_CHARACTERS = new HashSet<>(Arrays.asList(0x202A, 0x202B, 0x202C, 0x202D, 0x202E, 0x2066, 0x2067, 0x2068, 0x2069));
    private static final String ESCAPED_BIDI_PREFIX = "<span contenteditable=\"false\" class=\"bidi-unicode\" title=\"%s\" data-bidi-character-code=\"";
    private static final String ESCAPED_BIDI_MIDDLE_PART = "\">";
    private static final String ESCAPED_BIDI_SUFFIX = "<!--Bidi char--></span>";
    private final Set<Integer> bidiUnicodeCharacters;
    private final Supplier<String> warningMessageSupplier;
    public DefaultUnicodeEscapingTool(Supplier<String> warningMessageSupplier) {
        this(Collections.emptySet(), warningMessageSupplier);
    }

    public DefaultUnicodeEscapingTool(Set<Integer> bidiUnicodeCharacters, Supplier<String> warningMessageSupplier) {
        this.bidiUnicodeCharacters = new HashSet<>(bidiUnicodeCharacters);
        this.bidiUnicodeCharacters.addAll(DEFAULT_BIDI_CHARACTERS);
        this.warningMessageSupplier = warningMessageSupplier;
    }


    public String escapeBidiCharacters(String text) {
        StringBuilder builder = new StringBuilder(text.length());
        String prefix = String.format(ESCAPED_BIDI_PREFIX, warningMessageSupplier.get());
        text.codePoints()
                .forEach(character -> {
                    if (bidiUnicodeCharacters.contains(character)) {
                        builder.append(prefix);
                        builder.append(Integer.toHexString(character).toUpperCase(Locale.ROOT));
                        builder.append(ESCAPED_BIDI_MIDDLE_PART);
                        builder.append(Character.toChars(character));
                        builder.append(ESCAPED_BIDI_SUFFIX);
                    } else {
                        builder.append(Character.toChars(character)); //Character.toChars changes codePoint back to unicode characters
                    }
                });
        return builder.toString();
    }
}
