package com.atlassian.renderer;

import com.atlassian.renderer.v2.Replacer;

import java.util.regex.Pattern;

/**
 * Represents the different types of token that are used in the {@link RenderedContentStore} when rendering wiki markup
 * to html.
 */
public enum TokenType {
    /**
     * Inline tokens are used for inline content such as plain text, spans, images, bold tags, and links.
     */
    INLINE("inltokxyzkdtnhgnsbdfinltok"),
    /**
     * Block tokens are used for block elements such as paragraphs, divs, tables and pre blocks.
     */
    BLOCK("blktokxyzkdtnhgnsbdfblktok"),
    /**
     * Similar (but not identical) to the html concept of inline-block, these tokens are used for elements that
     * have a block and inline nature, as divs can.  In this case they are used for tokens that shouldn't be wrapped
     * in a paragraph, because they can be a block, but also shouldn't break a containing paragraph in half,
     * because they can be inline.
     */
    INLINE_BLOCK("ibltokxyzkdtnhgnsbdfibltok");

    private final String tokenString;
    private final String tokenPatternString;
    private final Pattern tokenPattern;
    private final Replacer tokenDeleteReplacer;

    TokenType(String tokenString) {
        this.tokenString = tokenString;
        this.tokenPatternString = tokenString + "\\d+" + tokenString;
        this.tokenPattern = Pattern.compile(tokenPatternString);
        this.tokenDeleteReplacer = new Replacer(tokenPattern, "");
    }

    /**
     * An unlikely-to-appear-in-real-text token, for use in content replacement.
     */
    public String getTokenMarker() {
        return tokenString;
    }

    /**
     * A String that can be used as to make {@link Pattern}s to find tokens of this type.  Intended for use in composing
     * more complex patterns, since {@link #getTokenPattern()} is also available.
     */
    public String getTokenPatternString() {
        return tokenPatternString;
    }

    /**
     * A Pattern that can be used to find tokens of this type.
     */
    public Pattern getTokenPattern() {
        return tokenPattern;
    }

    public Replacer getTokenDeleteReplacer() {
        return tokenDeleteReplacer;
    }
}
