package com.atlassian.renderer.wysiwyg;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.TokenType;
import com.atlassian.renderer.macro.RadeoxCompatibilityMacro;
import com.atlassian.renderer.util.NodeUtil;
import com.atlassian.renderer.util.RendererUtil;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.v2.components.HtmlEscaper;
import com.atlassian.renderer.v2.components.MacroRendererComponent;
import com.atlassian.renderer.v2.components.MacroTag;
import com.atlassian.renderer.v2.macro.Macro;
import com.atlassian.renderer.v2.macro.ResourceAwareMacroDecorator;
import com.atlassian.renderer.v2.macro.WysiwygBodyType;
import com.atlassian.renderer.v2.macro.basic.NoformatMacro;
import com.atlassian.renderer.wysiwyg.converter.DefaultWysiwygConverter;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
import org.w3c.dom.Node;

import java.util.HashMap;
import java.util.Map;

/**
 * Class responsible for the Rendering and Conversion of Macros, between wiki markup and wysiwyg xhtml.
 */
public class WysiwygMacroHelper {
    private static final Logger log = Logger.getLogger(WysiwygMacroHelper.class);

    public static final String MACRO_TAG_PARAM = "MACRO_TAG_PARAM";
    public static final String MACRO_CLASS = "wysiwyg-macro";
    private static final String MACRO_INLINE_CLASS = "wysiwyg-macro-inline";
    private static final String MACRO_BODY_CLASS = "wysiwyg-macro-body";
    private static final String MACRO_BODY_PREFORMAT_CLASS = "wysiwyg-macro-body-preformat";
    private static final String MACRO_TAG_CLASS = "wysiwyg-macro-tag";
    private static final String MACRO_START_TAG_CLASS = "wysiwyg-macro-starttag";
    private static final String MACRO_END_TAG_CLASS = "wysiwyg-macro-endtag";
    private static final String MACRO_BODY_BREAK_CLASS = "wysiwyg-macro-body-newline";

    public static final String MACRO_NAME_ATTRIBUTE = "macroname";
    public static final String MACRO_START_TAG_ATTRIBUTE = "macrostarttag";

    public static final String MACRO_HAS_BODY_ATTRIBUTE = "macrohasbody";

    public static final String MACRO_HAS_NEWLINE_BEFORE_BODY_ATTRIBUTE = "wikihasnewlinebeforebody";
    public static final String MACRO_HAS_NEWLINE_AFTER_BODY_ATTRIBUTE = "wikihasnewlineafterbody";
    public static final String MACRO_HAS_PRECEDING_NEWLINE_ATTRIBUTE = "wikihasprecedingnewline";
    public static final String MACRO_HAS_TRAILING_NEWLINE_ATTRIBUTE = "wikihastrailingnewline";

    private static final String NEW_LINE = "\n";

    private final MacroRendererComponent macroRendererComponent;
    private static final String CLASS_ATTRIBUTE_NAME = "class";
    /**
     * Class given to paragraphs whose purpose is to allow somewhere for the cursor to go after a macro.
     *
     * @deprecated use DefaultWysiwygConverter#PADDING_CLASS
     */
    public static final String MACRO_PADDING_CLASS = DefaultWysiwygConverter.CURSOR_PLACEMENT_CLASS;

    public WysiwygMacroHelper(MacroRendererComponent macroRendererComponent) {
        this.macroRendererComponent = macroRendererComponent;
    }

    /**
     * Renders a macro with the given information and outputs the XHTML into the given buffer.
     *
     * @param startTag the start tag of the macro
     * @param macro    the macro to render. Note that this could be null.
     * @param body     the wiki markup body of the macro. This can also be null.
     * @param params   a map of macro parameters
     * @param context  the render context in which the macro is rendered
     * @param buffer   the buffer to which the rendered output is appended
     */
    public void renderMacro(MacroTag startTag, Macro macro, String body, Map<String, Object> params, RenderContext context, StringBuffer buffer) {
        // Macros responsible for their own rendering, including the wrapping div/span elements.
        if (macro != null && macro.suppressSurroundingTagDuringWysiwygRendering()) {
            renderMacroResponsibleForOwnRendering(startTag, macro, body, params, context, buffer);
            // Inline macros don't get padding but all others do.
            switch (macro.getTokenType(params, body, context)) {
                case INLINE:
                    break;
                case INLINE_BLOCK:
                case BLOCK:
                    padForCursorPlacement(context, buffer);
                default:
                    log.warn("Unexpected tokenType:" + macro.getTokenType(params, body, context));
            }
            return;
        }

        // Macros that we need to create wrapping span/divs for
        renderWithSurroundingHtmlTag(startTag, macro, body, params, context, buffer);
    }

    /**
     * Render the macro with surrounding start and end html tags. All macros handled by this method have divs wrapped
     * around them, not spans, even if they are inline macros.
     *
     * @param startTag the start tag of the macro
     * @param macro    the macro to render. Note that this could be null.
     * @param body     the wiki markup body of the macro. This can also be null.
     * @param params   a map of macro parameters
     * @param context  the render context in which the macro is rendered
     * @param buffer   the buffer to which the rendered output is appended
     */
    private void renderWithSurroundingHtmlTag(MacroTag startTag, Macro macro, String body, Map<String, Object> params,
                                              RenderContext context, StringBuffer buffer) {
        StringBuffer result = new StringBuffer();

        result.append(createWrappingDivStartElement(startTag, macro, params, body, context));
        if (!requiresWikiTags(macro)) {
            renderWithoutWikiTags(startTag, macro, body, params, context, result);
        } else {
            renderWithWikiTags(startTag, macro, body, context, result);
        }
        result.append("</div>");

        final String resultToken;
//        resultToken = context.addRenderedContent(result.toString(), macro != null && macro.isInline());
        // surrounding html tags are displayed as divs even though the macros themselves will render inline.
        resultToken = context.addRenderedContent(result.toString(), TokenType.BLOCK);
        buffer.append(resultToken);

        padForCursorPlacement(context, buffer);
    }

    private boolean requiresWikiTags(Macro macro) {
        return macro == null || macro.suppressMacroRenderingDuringWysiwyg();
    }

    /**
     * Renders the macro with its start and end wiki markup tags.
     *
     * @see #convertWithWikiTags(NodeContext, DefaultWysiwygConverter, Macro, StringBuffer)
     */
    private void renderWithWikiTags(MacroTag startTag, Macro macro, String body, RenderContext context, StringBuffer result) {
        addStartMacroWikiMarkup(startTag, result);
        renderBody(startTag, macro, body, context, result);

        if (startTag.getEndTag() != null) {
            if (startTag.getEndTag().isNewlineBefore())
                result.append("<br class=\"" + MACRO_BODY_BREAK_CLASS + "\"/>");

            addEndMacroWikiMarkup(startTag, result);
        }
    }

    private void renderBody(MacroTag startTag, Macro macro, String body, RenderContext context, StringBuffer result) {
        if (!shouldRenderBody(startTag, macro, body))
            return;

        if (startTag.isNewlineAfter())
            result.append("<br class=\"" + MACRO_BODY_BREAK_CLASS + "\"/>");

        // Since we add br tags around the macro if there are newlines between the wiki tags and the body
        // those newlines should not be considered part of the body.
        body = RenderUtils.trimInitialNewline(body);
        body = StringUtils.chomp(body, "\n");

        if (macro != null && macro.getWysiwygBodyType() == WysiwygBodyType.PREFORMAT) {
            appendDivStartWithClasses(result, MACRO_BODY_CLASS, MACRO_BODY_PREFORMAT_CLASS,
                    NoformatMacro.PREFORMATTED_CONTENT_WRAPPER_CLASS);

            result.append("<pre>");
            RenderMode renderMode = RenderMode.allow(RenderMode.F_HTMLESCAPE);
            result.append((macroRendererComponent.getSubRenderer().render(body, context, renderMode)));
            result.append("</pre>");
            result.append("</div>");
        } else {
            appendDivStartWithClasses(result, MACRO_BODY_CLASS);

            RenderMode renderMode = RenderMode.ALL;
            result.append((macroRendererComponent.getSubRenderer().render(body, context, renderMode)));
            result.append("</div>");
        }
    }

    private boolean shouldRenderBody(MacroTag startTag, Macro macro, String body) {
        if (macro != null && !macro.hasBody()) {

            if (StringUtils.isNotEmpty(body) && log.isEnabledFor(Priority.WARN)) {
                // This indicates a logic error or a race condition. Two macro tags for a macro that takes
                // no body should be interpreted as two macros, not one with a body.
                log.warn(startTag.command + " macro declares it doesn't take a body, so ignoring: " + body);
            }

            return false;
        }
        if (macro == null && StringUtils.isEmpty(body)) {
            // unknown bodyless macro, not an error, but don't render a body.
            return false;
        }
        if (isRadeoxCompatibilityMacroOrDecoratingOne(macro) && StringUtils.isEmpty(body)) {
            // Some radeox compatibility macros return true from macro.hasBody() even though they usually don't.
            // We don't render these macros' bodies when no body is given (eg when there is no end tag found)
            return false;
        }
        return true;
    }

    private static void appendDivStartWithClasses(StringBuffer result, String... classNames) {
        result.append("<div class=\"");
        result.append(StringUtils.join(classNames, ' '));
        result.append("\">");
    }

    /**
     * Renders a macro without including the wiki markup tags. The macro is rendered as it would for
     * normal display mode.
     *
     * @see #convertWithoutWikiTags(NodeContext, DefaultWysiwygConverter, Macro, StringBuffer, String)
     */
    private void renderWithoutWikiTags(MacroTag startTag, Macro macro, String body, Map<String, Object> params,
                                       RenderContext context, StringBuffer result) {
        macroRendererComponent.processMacro(startTag.command, macro, body, params, context, result);
    }

    private void addStartMacroWikiMarkup(MacroTag startTag, StringBuffer buffer) {
        final String escapedStartTag = HtmlEscaper.escapeAll(startTag.originalText, false);
        appendDivStartWithClasses(buffer, MACRO_TAG_CLASS, MACRO_START_TAG_CLASS);
        buffer.append(escapedStartTag);
        buffer.append("</div>");
    }

    private void addEndMacroWikiMarkup(MacroTag macroTag, StringBuffer result) {
        final String escapedMacroTag = HtmlEscaper.escapeAll(macroTag.command, false);
        appendDivStartWithClasses(result, MACRO_TAG_CLASS, MACRO_END_TAG_CLASS);
        result.append("{").append(escapedMacroTag).append("}");
        result.append("</div>");
    }

    // We can remove this padding for renderWithSurroundingHtmlTag() if CONF-15225 is fixed.
    // This provides a place for users to add more text. The 'atl_conf_pad' class allows us to ignore it when
    // converting back to markup, if the user hasn't changed the content, BUT WE DON'T.
    private void padForCursorPlacement(RenderContext context, StringBuffer buffer) {
        buffer.append(context.addRenderedContent(DefaultWysiwygConverter.CURSOR_PLACEMENT_PARAGRAPH, TokenType.BLOCK));
    }

    private void renderMacroResponsibleForOwnRendering(MacroTag startTag, Macro macro, String body, Map<String, Object> params,
                                                       RenderContext context, StringBuffer buffer) {
        HashMap<String, Object> amendedParams = new HashMap<String, Object>(params);
        amendedParams.put(MACRO_TAG_PARAM, startTag);
        final String escapedMacroTag = HtmlEscaper.escapeAll(startTag.command, false);
        macroRendererComponent.processMacro(escapedMacroTag, macro, body, amendedParams, context, buffer);
    }

    /**
     * Helper method to add various attributes.  Those are the cases that wrap
     * the content of the macro in a big span or div.
     *
     * @param macroTag tag object to provide an html wrapper for.
     * @return the same buffer passed into the buffer argument for chaining purposes
     */
    private String createWrappingDivStartElement(MacroTag macroTag, Macro macro, Map<String, Object> params, String body, RenderContext context) {
        StringBuffer buffer = new StringBuffer("<div ");

        if (macro != null && macro.getTokenType(params, body, context) == TokenType.INLINE)
            RendererUtil.appendAttribute(CLASS_ATTRIBUTE_NAME, MACRO_CLASS + " " + MACRO_INLINE_CLASS, buffer);
        else
            RendererUtil.appendAttribute(CLASS_ATTRIBUTE_NAME, MACRO_CLASS, buffer);

        // add common attributes
        RendererUtil.appendAttribute(MACRO_NAME_ATTRIBUTE, HtmlEscaper.escapeAll(macroTag.command, false), buffer);
        if (!requiresWikiTags(macro)) {
            final String escapedStartTag = HtmlEscaper.escapeAll(macroTag.originalText, false);
            RendererUtil.appendAttribute(MACRO_START_TAG_ATTRIBUTE, escapedStartTag, buffer);
        }

        MacroTag endTag = macroTag.getEndTag();
        boolean hasBody = (macro != null && macro.hasBody()) || (macro == null && endTag != null);

        // add body attributes
        RendererUtil.appendAttribute(MACRO_HAS_BODY_ATTRIBUTE, hasBody, buffer);
        if (hasBody) {
            if (!requiresWikiTags(macro)) {
                RendererUtil.appendAttribute(MACRO_HAS_NEWLINE_BEFORE_BODY_ATTRIBUTE, macroTag.isNewlineAfter(), buffer);
                if (endTag != null) {
                    RendererUtil.appendAttribute(MACRO_HAS_NEWLINE_AFTER_BODY_ATTRIBUTE, endTag.isNewlineBefore(), buffer);
                }
            }
        }

        // add macro new line attributes
        boolean hasTrailingNewline = endTag == null ? macroTag.isNewlineAfter() : endTag.isNewlineAfter();
        RendererUtil.appendAttribute(MACRO_HAS_PRECEDING_NEWLINE_ATTRIBUTE, macroTag.isNewlineBefore(), buffer);
        RendererUtil.appendAttribute(MACRO_HAS_TRAILING_NEWLINE_ATTRIBUTE, hasTrailingNewline, buffer);

        buffer.append(">");
        return buffer.toString();
    }

    private static boolean isRadeoxCompatibilityMacroOrDecoratingOne(Macro macro) {
        if (macro instanceof RadeoxCompatibilityMacro)
            return true;

        if (macro instanceof ResourceAwareMacroDecorator) {
            Macro decoratedMacro = ((ResourceAwareMacroDecorator) macro).getMacro();
            return decoratedMacro instanceof RadeoxCompatibilityMacro;
        }
        return false;
    }

    /**
     * Retrieves the macro name from the given node.
     */
    public static String getMacroName(Node node) {
        return NodeUtil.getAttribute(node, MACRO_NAME_ATTRIBUTE);
    }

    /**
     * Returns true if the node is a macro body node.
     */
    public static boolean isMacroBody(Node node) {
        String classValue = NodeUtil.getAttribute(node, CLASS_ATTRIBUTE_NAME);
        return classValue != null && ArrayUtils.contains(classValue.split(" "), MACRO_BODY_CLASS);
    }

    /**
     * Returns true if the node is a macro body node.
     */
    public static boolean isMacroTag(Node node) {
        String classValue = NodeUtil.getAttribute(node, CLASS_ATTRIBUTE_NAME);
        return classValue != null && ArrayUtils.contains(classValue.split(" "), MACRO_TAG_CLASS);
    }

    /**
     * Converts a macro node back to wiki markup.
     *
     * @param nodeContext             the context the node is currently in
     * @param defaultWysiwygConverter the converter that is calling this method
     * @param macro                   the macro that is being converted
     * @return wiki markup string of the macro
     */
    public static String convertMacroFromNode(final NodeContext nodeContext, final DefaultWysiwygConverter defaultWysiwygConverter, final Macro macro) {
        StringBuffer result = new StringBuffer();

        if (shouldPadMacroWithPrecedingNewline(nodeContext))
            result.append(NEW_LINE);

        String startTagText = nodeContext.getAttribute(MACRO_START_TAG_ATTRIBUTE);
        if (startTagText != null)
            convertWithoutWikiTags(nodeContext, defaultWysiwygConverter, macro, result, startTagText);
        else
            convertWithWikiTags(nodeContext, defaultWysiwygConverter, macro, result);

        return result.toString();
    }

    private static boolean shouldPadMacroWithPrecedingNewline(NodeContext nodeContext) {
        // If we haven't recorded the need for padding, then we don't need it.
        if (!nodeContext.getBooleanAttributeValue(MACRO_HAS_PRECEDING_NEWLINE_ATTRIBUTE, true))
            return false;

        // If we are the first thing in our container, we leave prior padding to our container.
        final Node previousSibling = nodeContext.getPreviousSibling();
        if (previousSibling == null)
            return false;

        // If there's a user newline directly before, then we don't add extra padding
        return !DefaultWysiwygConverter.isUserNewline(previousSibling);
    }

    /**
     * Convert the node with the {@link #MACRO_START_TAG_ATTRIBUTE}.
     *
     * @see #renderWithoutWikiTags(MacroTag, Macro, String, Map, RenderContext, StringBuffer)
     */
    private static void convertWithoutWikiTags(NodeContext nodeContext, DefaultWysiwygConverter defaultWysiwygConverter,
                                               Macro macro, StringBuffer result, String startTagText) {
        boolean hasBody = nodeContext.getBooleanAttributeValue(MACRO_HAS_BODY_ATTRIBUTE, false);
        boolean hasNewLineBeforeBody = hasBody && nodeContext.getBooleanAttributeValue(MACRO_HAS_NEWLINE_BEFORE_BODY_ATTRIBUTE, true);
        boolean hasNewLineAfterBody = nodeContext.getBooleanAttributeValue(MACRO_HAS_NEWLINE_AFTER_BODY_ATTRIBUTE, true);

        result.append(startTagText);
        if (hasNewLineBeforeBody)
            result.append(NEW_LINE);

        if (!hasBody)
            return;

        String body = convertMacroBodyWithoutWikiTags(nodeContext, defaultWysiwygConverter, macro);
        if (body != null) {
            result.append(body);
            if (hasNewLineAfterBody)
                result.append(NEW_LINE);

            result.append("{").append(NodeUtil.getAttribute(nodeContext.getNode(), MACRO_NAME_ATTRIBUTE)).append("}");
        }
    }

    /**
     * Convert a macro node that was rendered with wiki tags around the body.
     *
     * @see #renderWithWikiTags(MacroTag, Macro, String, RenderContext, StringBuffer)
     */
    private static void convertWithWikiTags(NodeContext nodeContext, DefaultWysiwygConverter defaultWysiwygConverter,
                                            Macro macro, StringBuffer result) {
        // Unknown macros render with RenderMode.All, so we have to convert the children.
        if (macro == null) {
            result.append(convertMacroBody(nodeContext, defaultWysiwygConverter));
        }
        // Macros that handle their own rendering, so we don't need to convert the body here
        else if (macro.suppressSurroundingTagDuringWysiwygRendering()) {
            // do nothing
        } else if (!macro.hasBody()) // Then we only need the tag text.  Ignore RTE.
        {
            result.append(DefaultWysiwygConverter.getRawChildText(nodeContext.getNode(), true));
        }
        // Macros that provide their own method to convert their own *bodies*.
        else if (getMacroBodyConverter(macro) != null) {
            MacroBodyConverter macroBodyConverter = getMacroBodyConverter(macro);
            result.append(macroBodyConverter.convertXhtmlToWikiMarkup(nodeContext, defaultWysiwygConverter));
        } else {
            // Default to rendered body
            result.append(convertMacroBody(nodeContext, defaultWysiwygConverter));
        }
    }

    /**
     * Converts the macro body text from the node. This method includes new line characters before/after the body text.
     *
     * @return the macro body in wiki markup, can be null for some cases
     */
    private static String convertMacroBodyWithoutWikiTags(final NodeContext nodeContext, final DefaultWysiwygConverter defaultWysiwygConverter, final Macro macro) {
        // Macros that handle their own rendering, so we don't need to convert the body here
        if (macro.suppressSurroundingTagDuringWysiwygRendering()) {
            return null;
        }
        if (!macro.hasBody()) // shouldn't get here.
        {
            return null;
        }
        // Macros that provide their own method to convert their own *bodies*.
        MacroBodyConverter macroBodyConverter = getMacroBodyConverter(macro);
        if (macroBodyConverter != null) {
            return macroBodyConverter.convertXhtmlToWikiMarkup(nodeContext, defaultWysiwygConverter);
        }
        return convertMacroBody(nodeContext, defaultWysiwygConverter);
    }

    private static MacroBodyConverter getMacroBodyConverter(Macro macro) {
        if (macro instanceof ResourceAwareMacroDecorator) // Unwrap decorating macro
        {
            macro = ((ResourceAwareMacroDecorator) macro).getMacro();
        }
        if (macro instanceof MacroBodyConverter) {
            return (MacroBodyConverter) macro;
        }
        return null;
    }

    /**
     * Converts the children on the current node to text.  {@link #MACRO_TAG_CLASS} nodes are converted as raw text.
     * Other nodes are converted as html.
     */
    private static String convertMacroBody(final NodeContext nodeContext, final DefaultWysiwygConverter defaultWysiwygConverter) {
        Node node = nodeContext.getNode();
        StringBuffer wikiText = new StringBuffer();

        if (node != null && node.getChildNodes() != null) {
            NodeContext childContext = nodeContext.getFirstChildNodeContext();
            while (childContext != null) {
                if (isMacroTag(childContext.getNode()))
                    wikiText.append(DefaultWysiwygConverter.getRawChildText(childContext.getNode(), false));
                else
                    wikiText.append(defaultWysiwygConverter.convertNode(childContext));

                childContext = nodeContext.getNodeContextForNextChild(childContext);
            }
        }

        return RenderUtils.trimNewlinesAndEscapedNewlines(wikiText.toString());
    }
}
