package com.atlassian.renderer.v2.plugin;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.web.descriptors.WeightedDescriptor;
import com.atlassian.plugin.web.descriptors.WeightedDescriptorComparator;
import com.atlassian.renderer.v2.components.RendererComponent;
import com.atlassian.renderer.v2.components.TextConverter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * Component for managing the retrieval of renderer components from the plugin manager. This class is
 * used by the PluggableV2Renderer and the PluggableWysiwygConverter,
 * but has been extracted because the former class would be really tricky
 * to unit-test.
 */
class RendererComponentsAccessor {
    private static final Comparator<WeightedDescriptor> COMPARATOR = new WeightedDescriptorComparator();

    private final PluginAccessor pluginAccessor;

    RendererComponentsAccessor(PluginAccessor pluginAccessor) {
        this.pluginAccessor = pluginAccessor;
    }

    /**
     * Get the renderer components that are currently activated, in the order that they should
     * be applied.
     *
     * @return the current active renderer components
     */
    public List<RendererComponent> getActiveRendererComponents() {
        List<RendererComponentModuleDescriptor> moduleDescriptors = getEnabledRendererComponentModuleDescriptors();
        List<RendererComponent> components = new ArrayList<RendererComponent>(moduleDescriptors.size());

        for (RendererComponentModuleDescriptor descriptor : moduleDescriptors) {
            components.add(descriptor.getModule());
        }

        return components;
    }

    /**
     * Get the renderer components that are currently activated, and that implement
     * {@link com.atlassian.renderer.v2.components.TextConverter}, in the order that they should be applied.
     *
     * @return the current active renderer text converters
     */
    public List<TextConverter> getActiveTextConverterComponents() {
        List<RendererComponentModuleDescriptor> moduleDescriptors = getEnabledRendererComponentModuleDescriptors();
        List<TextConverter> converters = new ArrayList<TextConverter>(moduleDescriptors.size());

        for (RendererComponentModuleDescriptor descriptor : moduleDescriptors) {
            Object module = descriptor.getModule();
            if (module instanceof TextConverter) {
                converters.add((TextConverter) module);
            }
        }

        return converters;
    }

    private List<RendererComponentModuleDescriptor> getEnabledRendererComponentModuleDescriptors() {
        List<RendererComponentModuleDescriptor> modules = new ArrayList<RendererComponentModuleDescriptor>(
                pluginAccessor.getEnabledModuleDescriptorsByClass(RendererComponentModuleDescriptor.class));
        Collections.sort(modules, COMPARATOR);
        return modules;
    }
}
