package com.atlassian.renderer.v2.components.phrase;

import com.atlassian.renderer.IconManager;
import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.util.RegExpUtil;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.Replacer;
import com.atlassian.renderer.v2.components.AbstractRegexRendererComponent;
import com.atlassian.renderer.v2.components.TextConverter;
import com.atlassian.util.concurrent.LazyReference;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class EmoticonRendererComponent extends AbstractRegexRendererComponent implements TextConverter {
    public static final String MAX_EMOTICONS_TO_RENDER_SYSTEM_PROPERTY_KEY = "atlassian.renderer.max.emoticons";
    private static final Logger log = Logger.getLogger(EmoticonRendererComponent.class);
    private final int MAX_EMOTICONS = Integer.getInteger(MAX_EMOTICONS_TO_RENDER_SYSTEM_PROPERTY_KEY, Integer.MAX_VALUE);

    private final IconManager iconManager;
    private final LazyReference<Emoticons> emoticons;

    public EmoticonRendererComponent(final IconManager iconManager) {
        this.iconManager = iconManager;
        this.emoticons = new LazyReference<Emoticons>() {
            @Override
            protected Emoticons create() throws Exception {
                return new Emoticons(iconManager.getEmoticonSymbols());
            }
        };
    }

    public boolean shouldRender(RenderMode renderMode) {
        return renderMode.renderPhrases();
    }

    public String render(String wiki, RenderContext context) {
        return emoticons.get().render(wiki, context);
    }

    public String convertToWikiMarkup(String text) {
        return emoticons.get().toWikiMarkup(text);
    }

    public void appendSubstitution(StringBuffer buffer, RenderContext context, Matcher matcher) {
        String match = matcher.group(0);        // Emoticon is escaped.  Use literal emoticon
        if (match.startsWith("\\")) {
            buffer.append(match.substring(1));
        } else {
            buffer.append(iconManager.getEmoticon(match).toHtml(context.getImagePath()));
        }
    }

    // package protected so unit tests can override it, eliminating the need to rely on the manipulation of system properties
    int maxEmoticons() {
        return MAX_EMOTICONS;
    }

    /**
     * Immutable container class for Emoticon.
     */
    private class Emoticons {
        private final List<Emoticon> emoticons;
        /**
         * This pattern is dynamically constructed to replace emoticons in the original string
         * (up to a maximum of {@link #MAX_EMOTICONS} - defined per product)
         * As per JRA-36529, if we have more than N substitutions to perform, this can lead to OOME.
         */
        private final Pattern allEmoticonsPattern;

        public Emoticons(final String[] symbols) {
            final StringBuilder counterPattern = new StringBuilder();

            emoticons = new ArrayList<Emoticon>(symbols.length);
            for (String symbol : symbols) {
                String patternString = "(\\\\?" + RegExpUtil.convertToRegularExpression(symbol) + ')' + PhraseRendererComponent.VALID_END;
                emoticons.add(new Emoticon(Pattern.compile(patternString), symbol));
                counterPattern.append(patternString + "|");
            }

            // remove the extraneous |
            counterPattern.setLength(counterPattern.length() - 1);
            allEmoticonsPattern = Pattern.compile(counterPattern.toString());
        }

        public String render(String wiki, RenderContext context) {
            final Matcher matcher = allEmoticonsPattern.matcher(wiki);
            if (!matcher.find()) {
                return wiki;
            }

            int count = 1;
            int mark = 0;
            // as at least one emoticon will be replaced in the original String, we can double the initial capacity
            StringBuffer sb = new StringBuffer(wiki.length() * 2);
            do {
                sb.append(wiki, mark, matcher.start());
                appendSubstitution(sb, context, matcher);
                mark = matcher.end();
            }
            while (matcher.find() && ++count <= maxEmoticons());

            if (count > maxEmoticons()) {
                log.warn("Max number of emoticons (" + maxEmoticons() + ") reached for input hashed as " + DigestUtils.md2Hex(wiki) + ". MaxNumberOfEmoticonsToRenderReached");
            }

            return sb.append(wiki, mark, wiki.length()).toString();
        }

        public String toWikiMarkup(String text) {
            for (Emoticon emoticon : emoticons) {
                text = emoticon.toWiki(text);
            }
            return text;
        }
    }

    private class Emoticon {
        private final String symbol;
        private final Replacer replacer;

        private Emoticon(final Pattern pattern, final String symbol) {
            this.symbol = symbol;
            this.replacer = new Replacer(pattern, "\\\\$1", getSymbol());
        }

        public String getSymbol() {
            return symbol;
        }

        public String toWiki(String text) {
            return replacer.replaceAll(text);
        }
    }
}
