package com.atlassian.renderer.v2.components.block;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.TokenType;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.SubRenderer;
import com.atlassian.renderer.v2.components.RendererComponent;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * RendererComponent to handle block structures such as lists, tables and paragraphs.  This component is typically the
 * second component to run, the first being the MacroRendererComponent. The component essentially has two tasks:
 * invoking the various {@link BlockRenderer}s on the lines of wiki markup, and ensuring anything not dealt with by
 * a {@link BlockRenderer} is grouped into paragraphs appropriately.
 * <p>
 * The component has a configurable list of {@link BlockRenderer}s.  The wiki markup is split up and fed to the
 * BlockRenderers line by line.  They are passed the current {@link LineWalker} so that they can temporarily wrest
 * control from the BlockRendererComponent and process however many consecutive lines they want.  Once a BlockRenderer
 * renders a line, that line will not be passed into any other BlockRenderers.  Rather, the previous paragraph will be
 * finished, and the rendered block will be added after it.
 * <p>
 * Content that does not get rendered by a {@link BlockRenderer} has further processing applied.  Consecutive chunks of
 * content that contain no block tokens and are not handled by a BlockRenderer are grouped together into a paragraph.
 * They then have p tags wrapped around them.  If a line containing block tokens is not handled by any BlockRenderer,
 * then it will be split up around the block tokens.  The block tokens will not form part of a paragraph but paragraphs
 * will be made of text between block tokens.  Text before the first token will be appended to the previous paragraph.
 * Text after the last token will be used as the start of the next paragraph.
 */
public class BlockRendererComponent implements RendererComponent {
    /**
     * Pattern for a single block token surrounded by nothing but whitespace.
     */
    private static final Pattern BLOCK_AND_WHITESPACE_PATTERN = Pattern.compile("(\\s*" + TokenType.BLOCK.getTokenPatternString() + ")+\\s*", Pattern.MULTILINE);
    private static final Pattern INLINE_BLOCK_AND_WHITESPACE_PATTERN = Pattern.compile("(\\s*" + TokenType.INLINE_BLOCK.getTokenPatternString() + ")+\\s*", Pattern.MULTILINE);
    private static final Pattern SINGLE_LINE_PARA = Pattern.compile("\\s*[\\p{Alnum}&&[^PLhb]][^\n]*");
    private static final List<String> LIST_OF_SINGLE_EMPTY_STRING = Collections.singletonList("");

    private BlockRenderer[] blockRenderers;
    private SubRenderer subRenderer;

    public BlockRendererComponent(SubRenderer subRenderer, List<BlockRenderer> blockRenderers) {
        this.subRenderer = subRenderer;
        this.blockRenderers = blockRenderers.toArray(new BlockRenderer[blockRenderers.size()]);
    }

    public void setBlockRenderers(List<BlockRenderer> blockRenderers) {
        this.blockRenderers = blockRenderers.toArray(new BlockRenderer[blockRenderers.size()]);
    }

    public boolean shouldRender(RenderMode renderMode) {
        return renderMode.renderParagraphs();
    }

    public String render(String wiki, RenderContext context) {
        // Shortcut the really common case where we're rendering one line of something that definitely isn't
        // block-level markup. (Hopefully this helps performance a tad)
        if (SINGLE_LINE_PARA.matcher(wiki).matches() && !containsBlockTokens(wiki) && !containsInlineBlockTokens(wiki)) {
            return context.addRenderedContent(renderParagraph(true, context, wiki), TokenType.BLOCK);
        }

        // todo: These chunks of state and this method are dying to be refactored into a stateful object.
        LineWalker walker = new LineWalker(wiki);
        List<String> renderedLines = new ArrayList<String>();
        List<String> paragraph = new ArrayList<String>();
        boolean firstPara = true;

        while (walker.hasNext()) {
            String nextLine = walker.next();

            // Skip rendering of empty first paragraphs when "renderFirstParagraph" is off.
            // This is an overloading of the "renderFirstParagraph" flag and really needs to be changed.
            // Unfortunately that requires some substantial rebalancing of the whitespace preserving code
            if (firstPara && paragraph.isEmpty() && !context.getRenderMode().renderFirstParagraph() && nextLine.trim().length() == 0) {
                firstPara = false;
                continue;
            }
            String rendered = null;

            if (isOneBlockToken(nextLine)) {
                rendered = nextLine;
            } else if (isOneInlineBlockToken(nextLine)) {
                rendered = nextLine;
            } else if (!SINGLE_LINE_PARA.matcher(wiki).matches()) {
                rendered = applyBlockRenderers(context, walker, nextLine, rendered);
            }

            if (rendered == null) {
                // If no BlockRenderer dealt with the line, then we will deal with it as plain text, except that
                // block tokens must be split out, since other blocks (eg from macros) should never be wrapped in
                // paragraphs.  A p tag or pre tag or various other block tags within a p tag is invalid html and
                // can cause misrendering in some browsers (particularly TinyMCE3 + IE).
                for (String linePortion : splitLineByBlockTokens(nextLine)) {
                    if (!TokenType.BLOCK.getTokenPattern().matcher(linePortion).matches()) {
                        paragraph.add(linePortion);
                    } else {
                        flushParagraph(renderedLines, paragraph, context, firstPara);
                        renderedLines.add(linePortion);
                        firstPara = false;
                    }
                }
            } else {
                flushParagraph(renderedLines, paragraph, context, firstPara);
                renderedLines.add(rendered);
                firstPara = false;
            }
        }

        flushParagraph(renderedLines, paragraph, context, firstPara);
        return context.addRenderedContent(StringUtils.join(renderedLines.iterator(), "\n"), TokenType.BLOCK);
    }

    private static boolean isOneBlockToken(String string) {
        return BLOCK_AND_WHITESPACE_PATTERN.matcher(string).matches();
    }

    private static boolean isOneInlineBlockToken(String string) {
        return INLINE_BLOCK_AND_WHITESPACE_PATTERN.matcher(string).matches();
    }

    /**
     * Determine whether {@param string} contains any block tokens.
     */
    private static boolean containsBlockTokens(String string) {
        return TokenType.BLOCK.getTokenPattern().matcher(string).find();
    }

    /**
     * Determine whether {@param string} contains any inline block tokens.
     */
    private static boolean containsInlineBlockTokens(String string) {
        return TokenType.INLINE_BLOCK.getTokenPattern().matcher(string).find();
    }

    private String applyBlockRenderers(RenderContext context, LineWalker walker, String nextLine, String rendered) {
        for (BlockRenderer blockRenderer : blockRenderers) {
            rendered = blockRenderer.renderNextBlock(nextLine, walker, context, subRenderer);
            if (rendered != null)
                break;
        }
        return rendered;
    }

    /**
     * Split a line of text by its block tokens.
     *
     * @param line single line of text that may contain block tokens see
     *             {@link TokenType#getTokenPatternString()}
     * @return list of segments of line.  If line contains no block tokens then it will be return unsplit. Otherwise
     * the list will contain segments with no block tokens and segments that consist of exactly one block token.
     */
    static List<String> splitLineByBlockTokens(String line) {
        if (line.isEmpty()) {
            return LIST_OF_SINGLE_EMPTY_STRING;
        }

        final Matcher matcher = TokenType.BLOCK.getTokenPattern().matcher(line);
        if (!matcher.find()) {
            return Collections.singletonList(line);
        }

        final List<String> result = new ArrayList<String>();
        int mark = 0;
        do {
            final int start = matcher.start();
            if (start > mark) {
                result.add(line.substring(mark, matcher.start()));
            }
            result.add(matcher.group());
            mark = matcher.end();
        }
        while (matcher.find());

        if (line.length() > mark) {
            result.add(line.substring(mark));
        }
        return result;
    }

    private void flushParagraph(List<String> renderedLines, List<String> remainderedLines, RenderContext context, boolean firstParagraph) {
        if (remainderedLines.isEmpty()) {
            return;
        }

        String paragraph = StringUtils.join(remainderedLines.iterator(), "\n");
        renderedLines.add(renderParagraph(firstParagraph, context, paragraph));
        remainderedLines.clear();
    }

    private String renderParagraph(boolean firstParagraph, RenderContext context, String paragraph) {
        final String inner = subRenderer.render(paragraph, context, context.getRenderMode().and(RenderMode.INLINE));
        if (firstParagraph && !context.getRenderMode().renderFirstParagraph()) {
            return inner;
        }
        return "<p>" + inner + "</p>";
    }
}
