package com.atlassian.refapp.ctk.sal;

import com.atlassian.functest.junit.SpringAwareTestCase;
import com.atlassian.refapp.ctk.AppSpecificInfoProvider;
import com.atlassian.sal.api.features.DarkFeatureManager;
import com.atlassian.sal.api.features.EnabledDarkFeatures;
import com.atlassian.sal.api.features.InvalidFeatureKeyException;
import com.atlassian.sal.api.user.UserKey;
import org.junit.After;
import org.junit.Test;
import org.junit.internal.AssumptionViolatedException;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

public class DarkFeaturesTest extends SpringAwareTestCase
{
    private static final String ENABLED_FEATURE = "foo";
    private static final String DISABLED_FEATURE = "bar";
    private static final String INVALID_FEATURE_KEY = "invalid feature key";
    private static final String ADMIN_USER_FEATURE = "launch-button.feature";
    private static final UserKey ANONYMOUS = null;

    private DarkFeatureManager darkFeatureManager;
    private AppSpecificInfoProvider infoProvider;

    public void setDarkFeatureManager(DarkFeatureManager darkFeatureManager)
    {
        this.darkFeatureManager = darkFeatureManager;
    }

    public void setInfoProvider(AppSpecificInfoProvider infoProvider)
    {
        this.infoProvider = infoProvider;
    }

    @After
    public void disableAdminFeature() throws Exception
    {
        if (darkFeatureManager.isFeatureEnabledForUser(adminUser(), ADMIN_USER_FEATURE))
        {
            darkFeatureManager.disableFeatureForUser(adminUser(), ADMIN_USER_FEATURE);
        }
    }

    @Test
    public void systemFeatureEnabledForAllUsers()
    {
        assertFeatureEnabled(darkFeatureManager.isFeatureEnabledForAllUsers(ENABLED_FEATURE));
    }

    @Test
    public void systemFeatureDisabledForAllUsers()
    {
        assertFeatureDisabled(darkFeatureManager.isFeatureEnabledForAllUsers(DISABLED_FEATURE));
    }

    @Test
    public void invalidFeatureKeyIsNeverEnabledForAllUsers()
    {
        assertInvalidFeatureDisabled(darkFeatureManager.isFeatureEnabledForAllUsers(INVALID_FEATURE_KEY));
    }

    @Test
    public void systemFeatureEnabledForAnonymous()
    {
        assertFeatureEnabled(darkFeatureManager.isFeatureEnabledForUser(ANONYMOUS, ENABLED_FEATURE));
    }

    @Test
    public void systemFeatureDisabledForAnonymous()
    {
        assertFeatureDisabled(darkFeatureManager.isFeatureEnabledForUser(ANONYMOUS, DISABLED_FEATURE));
    }

    @Test
    public void invalidFeatureKeyIsNeverEnabledForAnonymous()
    {
        assertInvalidFeatureDisabled(darkFeatureManager.isFeatureEnabledForUser(ANONYMOUS, INVALID_FEATURE_KEY));
    }

    @Test
    public void systemFeatureEnabledForAuthenticatedUser()
    {
        assertFeatureEnabled(darkFeatureManager.isFeatureEnabledForUser(adminUser(), ENABLED_FEATURE));
    }

    @Test
    public void systemFeatureDisabledForAuthenticatedUser()
    {
        assertFeatureDisabled(darkFeatureManager.isFeatureEnabledForUser(adminUser(), DISABLED_FEATURE));
    }

    @Test
    public void invalidFeatureKeyIsNeverEnabledForAuthenticatedUser()
    {
        assertInvalidFeatureDisabled(darkFeatureManager.isFeatureEnabledForUser(adminUser(), INVALID_FEATURE_KEY));
    }

    @Test(expected = IllegalArgumentException.class)
    public void featureUndefinedForInvalidUser()
    {
        darkFeatureManager.isFeatureEnabledForUser(notExistingUser(), ENABLED_FEATURE);
    }

    @Test
    public void systemFeatureEnabledForCurrentUser()
    {
        assertFeatureEnabled(darkFeatureManager.isFeatureEnabledForCurrentUser(ENABLED_FEATURE));
    }

    @Test
    public void systemFeatureDisabledForCurrentUser()
    {
        assertFeatureDisabled(darkFeatureManager.isFeatureEnabledForCurrentUser(DISABLED_FEATURE));
    }

    @Test
    public void invalidFeatureKeyIsNeverEnabledForCurrentUser()
    {
        assertInvalidFeatureDisabled(darkFeatureManager.isFeatureEnabledForCurrentUser(INVALID_FEATURE_KEY));
    }

    @Test
    public void enableFeaturesForAllUsersContainsEnabledFeature()
    {
        assertFeatureEnabled(darkFeatureManager.getFeaturesEnabledForAllUsers());
    }

    @Test
    public void enabledFeaturesForAllUsersNotContainsDisabledFeature()
    {
        assertFeatureDisabled(darkFeatureManager.getFeaturesEnabledForAllUsers());
    }

    @Test
    public void enabledFeaturesForAnonymousContainsEnabledFeature()
    {
        assertFeatureEnabled(darkFeatureManager.getFeaturesEnabledForUser(null));
    }

    @Test
    public void enabledFeaturesForAnonymousNotContainsDisabledFeature()
    {
        assertFeatureDisabled(darkFeatureManager.getFeaturesEnabledForUser(null));
    }

    @Test
    public void enabledFeaturesForAuthenticatedUserContainsEnabledFeature()
    {
        assertFeatureEnabled(darkFeatureManager.getFeaturesEnabledForUser(adminUser()));
    }

    @Test
    public void enabledFeaturesForAuthenticatedUserNotContainsDisabledFeature()
    {
        assertFeatureDisabled(darkFeatureManager.getFeaturesEnabledForUser(adminUser()));
    }

    @Test(expected = IllegalArgumentException.class)
    public void enabledFeaturesForInvalidUser()
    {
        darkFeatureManager.getFeaturesEnabledForUser(notExistingUser());
    }

    @Test
    public void enabledFeaturesForCurrentUserContainsEnabledFeature()
    {
        assertFeatureEnabled(darkFeatureManager.getFeaturesEnabledForCurrentUser());
    }

    @Test
    public void enabledFeaturesForCurrentUserNotContainsDisabledFeature()
    {
        assertFeatureDisabled(darkFeatureManager.getFeaturesEnabledForCurrentUser());
    }

    @Test(expected = InvalidFeatureKeyException.class)
    public void invalidFeatureKeyNotEnabledForAllUsers()
    {
        darkFeatureManager.enableFeatureForAllUsers(INVALID_FEATURE_KEY);
    }

    @Test(expected = InvalidFeatureKeyException.class)
    public void invalidFeatureKeyNotDisabledForAllUsers()
    {
        darkFeatureManager.disableFeatureForAllUsers(INVALID_FEATURE_KEY);
    }

    @Test
    public void invalidFeatureKeyNotEnabledForAuthenticatedUser()
    {
        try
        {
            darkFeatureManager.enableFeatureForUser(adminUser(), INVALID_FEATURE_KEY);
            fail("Expected an " + InvalidFeatureKeyException.class.getName());
        }
        catch (UnsupportedOperationException e)
        {
            ignoreTest();
        }
        catch (InvalidFeatureKeyException e)
        {
            // success
        }
    }

    @Test
    public void featureNotEnabledForNotExisingUser()
    {
        try
        {
            darkFeatureManager.enableFeatureForUser(notExistingUser(), ADMIN_USER_FEATURE);
            fail("Expected an " + IllegalArgumentException.class.getName());
        }
        catch (UnsupportedOperationException e)
        {
            ignoreTest();
        }
        catch (IllegalArgumentException e)
        {
            // success
        }
    }

    @Test
    public void invalidFeatureKeyNotDisabledForAuthenticatedUser()
    {
        try
        {
            darkFeatureManager.disableFeatureForUser(adminUser(), INVALID_FEATURE_KEY);
            fail("Expected an " + InvalidFeatureKeyException.class.getName());
        }
        catch (UnsupportedOperationException e)
        {
            ignoreTest();
        }
        catch (InvalidFeatureKeyException e)
        {
            // success
        }
    }

    @Test
    public void featureNotDisabledForNotExisingUser()
    {
        try
        {
            darkFeatureManager.disableFeatureForUser(notExistingUser(), ADMIN_USER_FEATURE);
            fail("Expected an " + IllegalArgumentException.class.getName());
        }
        catch (UnsupportedOperationException e)
        {
            ignoreTest();
        }
        catch (IllegalArgumentException e)
        {
            // success
        }
    }

    @Test
    public void invalidFeatureKeyNotEnabledForCurrentUser()
    {
        try
        {
            darkFeatureManager.enableFeatureForCurrentUser(INVALID_FEATURE_KEY);
            fail("Expected an " + InvalidFeatureKeyException.class.getName());
        }
        catch (UnsupportedOperationException e)
        {
            ignoreTest();
        }
        catch (InvalidFeatureKeyException e)
        {
            // success
        }
        catch (IllegalStateException e)
        {
            //oddly enough success as well - the user is anonymous so should throw an IllegalStateException
        }
    }

    @Test
    public void invalidFeatureKeyNotDisabledForCurrentUser()
    {
        try
        {
            darkFeatureManager.disableFeatureForCurrentUser(INVALID_FEATURE_KEY);
            fail("Expected an " + InvalidFeatureKeyException.class.getName());
        }
        catch (UnsupportedOperationException e)
        {
            ignoreTest();
        }
        catch (InvalidFeatureKeyException e)
        {
            // success
        }
        catch (IllegalStateException e)
        {
          //oddly enough success as well - the user is anonymous so should throw an IllegalStateException
        }
    }

    @Test
    public void changeFeatureForAuthenticatedUser()
    {
        try
        {
            darkFeatureManager.enableFeatureForUser(adminUser(), ADMIN_USER_FEATURE);
        }
        catch (UnsupportedOperationException e)
        {
            ignoreTest();
        }

        assertTrue("Feature key '" + ADMIN_USER_FEATURE + "' must be enabled for the admin user", darkFeatureManager.isFeatureEnabledForUser(adminUser(), ADMIN_USER_FEATURE));
        assertFalse("Feature key '" + ADMIN_USER_FEATURE + "' must be disabled for all users", darkFeatureManager.isFeatureEnabledForAllUsers(ADMIN_USER_FEATURE));

        darkFeatureManager.disableFeatureForUser(adminUser(), ADMIN_USER_FEATURE);

        assertFalse("Feature key '" + ADMIN_USER_FEATURE + "' must be disabled for the admin user", darkFeatureManager.isFeatureEnabledForUser(adminUser(), ADMIN_USER_FEATURE));
    }

    private void assertFeatureEnabled(final boolean isEnabled)
    {
        assertTrue("Feature key '" + ENABLED_FEATURE + "' must be enabled", isEnabled);
    }

    private void assertFeatureEnabled(final EnabledDarkFeatures enabledDarkFeatures)
    {
        assertTrue("Feature key '"  + ENABLED_FEATURE + "' must be enabled; found: " + enabledDarkFeatures, enabledDarkFeatures.isFeatureEnabled(ENABLED_FEATURE));
    }

    private void assertFeatureDisabled(final boolean isEnabled)
    {
        assertFalse("Feature key " + DISABLED_FEATURE + "' must be disabled", isEnabled);
    }

    private void assertFeatureDisabled(final EnabledDarkFeatures enabledDarkFeatures)
    {
        assertFalse("Feature key '" + DISABLED_FEATURE + "' must be disabled; found: " + enabledDarkFeatures, enabledDarkFeatures.isFeatureEnabled(DISABLED_FEATURE));
    }

    private void assertInvalidFeatureDisabled(final boolean isEnabled)
    {
        assertFalse("Invalid feature key must never be enabled.", isEnabled);
    }

    private UserKey adminUser()
    {
        return infoProvider.getAdminUserKey();
    }

    private UserKey notExistingUser()
    {
        return new UserKey("somenonexistinguser");
    }

    private void ignoreTest()
    {
        throw new AssumptionViolatedException("The current application doesn't implement that feature");
    }
}
