import contextPath from 'wrm/context-path'; // eslint-disable-line
import PerformanceAPI from 'shims/performance-api';
import ConnectivityResults from './connectivity-results';
import { warn } from '../shims/logger';

const ASSET_DOWNLOAD_PATH = '/s/NOCACHE/_/download/resources/';

function fetchTestImage(baseUrl) {
    return new Promise((resolve, reject) => {
        const IMG_TEST_RESOURCE =
            'com.atlassian.plugins.static-assets-url:health-checks-test/health-checks/test-img.png';
        const nocache = new Date().getTime();
        const src = `${baseUrl}${ASSET_DOWNLOAD_PATH}${IMG_TEST_RESOURCE}?_=${nocache}`;

        const img = document.createElement('img');
        img.onload = function imgOnLoad() {
            resolve(src);
        };
        img.onerror = function imgOnError() {
            reject(new Error('Downloading the image failed'));
        };
        img.src = src;
    });
}

function fetchTestResources(baseUrl) {
    return Promise.all([fetchTestImage(baseUrl)]);
}

function getBaseUrl() {
    return `${window.location.protocol}//${window.location.host}${contextPath()}`;
}

/**
 * Verifies given resources were served over http2.
 * @param {String[]} testResourceUrls list of urls to verify
 * @param {Boolean} verifyDocument check the page itself (HTML, 'navigation' type)
 * @returns {Boolean|undefined} true if resources were served over http2, false otherwise
 *                              undefined when the result is unknown (not supported browser, error occurred)
 */
function areTestResourcesServedOverHttp2(testResourceUrls, verifyDocument = false) {
    if (!PerformanceAPI || 'getEntriesByType' in PerformanceAPI === false) {
        return undefined;
    }

    const resourcesInfo = PerformanceAPI.getEntriesByType('resource');
    if (!resourcesInfo[0] || 'nextHopProtocol' in resourcesInfo[0] === false) {
        return undefined;
    }

    const filteredInfo = resourcesInfo.filter(info => testResourceUrls.includes(info.name));
    if (filteredInfo.length !== testResourceUrls.length) {
        return undefined;
    }

    if (verifyDocument) {
        const docRequestInfo = PerformanceAPI.getEntriesByType('navigation')[0];
        if (!docRequestInfo) {
            return undefined;
        }
        filteredInfo.push(docRequestInfo);
    }

    return filteredInfo.reduce((acc, requestInfo) => {
        return acc && requestInfo.nextHopProtocol === 'h2';
    }, true);
}

function performConnectivityChecks(baseUrl, verifyDocument = false) {
    return new Promise(resolve => {
        const results = new ConnectivityResults();

        fetchTestResources(baseUrl)
            .then(testResourceUrls => {
                // fetching test resources on the instance succeeded
                results.isReachable = true;
                results.canServeAssets = true;
                results.http2 = areTestResourcesServedOverHttp2(testResourceUrls, verifyDocument);
            })
            .catch(e => {
                // fetching test resources on the instance failed
                // unable to verify http2
                results.isReachable = false;
                results.canServeAssets = false;
                warn(e);
            })
            .finally(() => resolve(results));
    });
}

export function performConnectivityChecksForTheInstance() {
    return performConnectivityChecks(getBaseUrl(), true);
}
export function performConnectivityChecksForCDN(cdnUrl) {
    return performConnectivityChecks(cdnUrl + contextPath());
}
export { getBaseUrl };
