import React, { useState } from 'react';
import Button, { ButtonGroup } from '@atlaskit/button';
import Form, { Field, ErrorMessage } from '@atlaskit/form';
import { RadioGroup } from '@atlaskit/radio';
import TextField from '@atlaskit/textfield';
import Tooltip from '@atlaskit/tooltip';

import AuiFlag from 'shims/aui-flag';
import formatter from 'shims/i18n';
import { saveCdnSettings } from 'admin-config/cdn-data-service';
import CdnData from 'admin-config/cdn-data';
import { useCdnData } from 'admin-config/cdn-data-context';
import { getBaseUrl } from 'health-checks/health-checks';
import ConnectivityHealthChecks, { getConnectivityResults } from 'health-checks/health-checks.ui';

import { CdnSettingsForm, FormFooter } from './settings-tab.styled';

const AdminSettingsTab = () => {
    const formId = 'cdn-admin-settings-form';
    const CDN_STATUS = { ON: 'on', OFF: 'off' };

    const [initCdnData, dispatch] = useCdnData();
    const [isCdnUrlValidated, setCdnUrlValidated] = useState(false);
    const [connectivityResults, setConnectivityResults] = useState(null);
    const [isValidatingCdnUrl, setValidatingCdnUrl] = useState(false);
    const [isCdnReachable, setCdnReachable] = useState(false);
    const [cdnToBeEnabled, setCdnToBeEnabled] = useState(initCdnData && initCdnData.enabled);
    // currently, there's no way to reset AK form, that's why "Form key={}" is used to reset the form
    const [resetId, setResetId] = useState(0);

    const CdnUrlValidation = {
        HTTPS: formatter.I18n.getText('static.assets.admin.settings.cdn.url.validation.https'),
        DOMAIN: formatter.I18n.getText('static.assets.admin.settings.cdn.url.validation.domain'),
    };

    function onReset() {
        setResetId(resetId + 1);
    }

    function validateCdnUrl(value) {
        if (value && !value.startsWith('https://')) {
            return CdnUrlValidation.HTTPS;
        }
        if (value && value.replace(/\/+$/, '') === getBaseUrl()) {
            return CdnUrlValidation.DOMAIN;
        }
        return undefined;
    }

    function getTooltipForValidateButton(cdnUrl) {
        return !cdnUrl
            ? formatter.I18n.getText(
                  'static.assets.admin.settings.validate.button.tooltip.on.empty.cdn.url'
              )
            : '';
    }

    function getTooltipForSubmitButton(isCdnToBeEnabled, isValidated, cannotReachCdn) {
        if (isCdnToBeEnabled) {
            if (!isValidated) {
                return formatter.I18n.getText(
                    'static.assets.admin.settings.save.button.tooltip.on.not.validated.cdn.url'
                );
            }
            if (cannotReachCdn) {
                return formatter.I18n.getText(
                    'static.assets.admin.settings.save.button.tooltip.on.connection.error'
                );
            }
        }
        return '';
    }

    function validateCdnConnection(cdnUrl) {
        setValidatingCdnUrl(true);

        getConnectivityResults(cdnUrl).then(results => {
            setValidatingCdnUrl(false);

            setConnectivityResults(results);
            setCdnUrlValidated(true);

            // make a judgment based on the results whether to allow to save settings or not
            setCdnReachable(results.cdn.canServeAssets);
        });
    }

    function onSubmit(data) {
        return new Promise((resolve, reject) => {
            const form = document.getElementById(formId);
            const isValidated = form && form.getAttribute('data-validated') === 'true';
            if (data['cdn-enabled'] === CDN_STATUS.ON && !isValidated) {
                reject(new Error('Form should be validated first'));
                return;
            }

            setCdnUrlValidated(false);

            saveCdnSettings(new CdnData(data['cdn-enabled'] === CDN_STATUS.ON, data['cdn-url']))
                .then(cdnData => {
                    AuiFlag({
                        title: cdnData.enabled
                            ? formatter.I18n.getText(
                                  'static.assets.admin.settings.cdn.enabled.success'
                              )
                            : formatter.I18n.getText(
                                  'static.assets.admin.settings.cdn.disabled.success'
                              ),
                        body: cdnData.enabled
                            ? formatter.I18n.getText(
                                  'static.assets.admin.settings.cdn.enabled.success.desc'
                              )
                            : formatter.I18n.getText(
                                  'static.assets.admin.settings.cdn.disabled.success.desc'
                              ),
                        type: 'success',
                        close: 'auto',
                    });

                    dispatch({ type: 'changeCdnData', cdnData });
                    resolve();
                })
                .catch(err => {
                    AuiFlag({
                        title: formatter.I18n.getText(
                            'static.assets.admin.settings.cdn.generic.error'
                        ),
                        body: formatter.I18n.getText(
                            'static.assets.admin.settings.cdn.generic.error.desc'
                        ),
                        type: 'warning',
                    });
                    reject(new Error('Something went wrong when saving the form', err));
                });
        });
    }

    return (
        <CdnSettingsForm>
            {initCdnData ? (
                <Form onSubmit={onSubmit} key={resetId}>
                    {({ formProps, submitting }) => (
                        <form
                            {...formProps}
                            action=""
                            id={formId}
                            data-validated={isCdnUrlValidated}
                            onChange={() => setCdnUrlValidated(false)}
                            onReset={onReset}
                        >
                            <Field
                                name="cdn-enabled"
                                label={formatter.I18n.getText(
                                    'static.assets.admin.settings.cdn.toggle.label'
                                )}
                                defaultValue={initCdnData.enabled ? CDN_STATUS.ON : CDN_STATUS.OFF}
                                isRequired
                            >
                                {({ fieldProps }) =>
                                    setCdnToBeEnabled(fieldProps.value === CDN_STATUS.ON) || (
                                        <RadioGroup
                                            {...fieldProps}
                                            options={[
                                                {
                                                    name: 'cdn-enabled',
                                                    value: CDN_STATUS.ON,
                                                    label: formatter.I18n.getText(
                                                        'static.assets.admin.settings.cdn.toggle.on'
                                                    ),
                                                },
                                                {
                                                    name: 'cdn-enabled',
                                                    value: CDN_STATUS.OFF,
                                                    label: formatter.I18n.getText(
                                                        'static.assets.admin.settings.cdn.toggle.off'
                                                    ),
                                                },
                                            ]}
                                        />
                                    )
                                }
                            </Field>

                            <Field
                                name="cdn-url"
                                label={formatter.I18n.getText(
                                    'static.assets.admin.settings.cdn.url.label'
                                )}
                                defaultValue={initCdnData.url ? initCdnData.url : ''}
                                isRequired={cdnToBeEnabled}
                                validate={validateCdnUrl}
                            >
                                {({ fieldProps, error }) => (
                                    <>
                                        <TextField
                                            placeholder={formatter.I18n.getText(
                                                'static.assets.admin.settings.cdn.url.placeholder'
                                            )}
                                            elemAfterInput={
                                                <Tooltip
                                                    content={getTooltipForValidateButton(
                                                        fieldProps.value
                                                    )}
                                                >
                                                    <Button
                                                        onClick={() =>
                                                            validateCdnConnection(fieldProps.value)
                                                        }
                                                        isDisabled={!fieldProps.value}
                                                        isLoading={isValidatingCdnUrl}
                                                    >
                                                        {formatter.I18n.getText(
                                                            'static.assets.admin.settings.validate.button'
                                                        )}
                                                    </Button>
                                                </Tooltip>
                                            }
                                            {...fieldProps}
                                            isInvalid={
                                                error || (isCdnUrlValidated && !isCdnReachable)
                                            }
                                        />
                                        {error && <ErrorMessage>{error}</ErrorMessage>}
                                    </>
                                )}
                            </Field>
                            {isCdnUrlValidated && (
                                <ConnectivityHealthChecks checks={connectivityResults} />
                            )}

                            <FormFooter>
                                <ButtonGroup>
                                    <Tooltip
                                        content={getTooltipForSubmitButton(
                                            cdnToBeEnabled,
                                            isCdnUrlValidated,
                                            !isCdnReachable
                                        )}
                                    >
                                        <Button
                                            appearance="primary"
                                            id="cdn-settings-submit"
                                            type="submit"
                                            isDisabled={
                                                submitting ||
                                                !(
                                                    !cdnToBeEnabled ||
                                                    (isCdnUrlValidated && isCdnReachable)
                                                )
                                            }
                                        >
                                            {formatter.I18n.getText(
                                                'static.assets.admin.settings.cdn.save.button'
                                            )}
                                        </Button>
                                    </Tooltip>
                                    <Button
                                        appearance="subtle"
                                        id="cdn-settings-cancel"
                                        type="reset"
                                    >
                                        {formatter.I18n.getText(
                                            'static.assets.admin.settings.cdn.cancel'
                                        )}
                                    </Button>
                                </ButtonGroup>
                            </FormFooter>
                        </form>
                    )}
                </Form>
            ) : (
                <section className="loading-container">
                    <aui-spinner size="medium" />
                </section>
            )}
        </CdnSettingsForm>
    );
};

export default AdminSettingsTab;
