import contextPath from 'wrm/context-path'; // eslint-disable-line
import { warn } from 'shims/logger';

const NETWORK_OVERHEAD_THRESHOLD = 1000;

function fetchNetworkStatistics() {
    return fetch(`${contextPath()}/rest/static-asset-caching/network-statistics`)
        .then(response => {
            if (response.ok) {
                return response.json();
            }
            throw new Error('Fetch failed');
        })
        .then(serverResponse => {
            if (!serverResponse || !Array.isArray(serverResponse.recentTransferCosts)) {
                throw new Error('Malformed network statistics received');
            }
            return serverResponse;
        });
}

/**
 * @returns {Promise<Number|string>} % of server responses
 *          with a network overhead of more than 1 second;
 *          range 0-100 or "-" in case of no data or error
 */
export function getNetworkOverheadMetric() {
    return new Promise(resolve => {
        fetchNetworkStatistics()
            .then(({ recentTransferCosts }) => {
                const total = recentTransferCosts.length;
                if (total === 0) {
                    resolve(0);
                    return;
                }

                const aboveThreshold = recentTransferCosts.reduce(
                    (acc, current) => (current > NETWORK_OVERHEAD_THRESHOLD ? acc + 1 : acc),
                    0
                );
                const perc = Math.round((100 * aboveThreshold) / total);

                resolve(perc);
            })
            .catch(err => {
                warn('Getting network overhead metric failed', err);
                resolve('-');
            });
    });
}
