import {
    getNetworkOverheadMetric,
    __RewireAPI__ as Rewire,
} from 'admin-config/performance/latency-metric-service';
import { mockFetchFailure, mockFetchSuccess, restoreFetch } from '../../fetch';

describe('getNetworkOverheadMetric', () => {
    function mockFetchNetworkStatistics(returnValue) {
        const mock = jest.fn(() => returnValue);
        Rewire.__Rewire__('fetchNetworkStatistics', mock);
        return mock;
    }

    afterEach(() => {
        Rewire.__ResetDependency__('fetchNetworkStatistics');
    });

    test.each`
        recentTransferCosts     | expectedMetric
        ${[]}                   | ${0}
        ${[800]}                | ${0}
        ${[1000]}               | ${0}
        ${[1100]}               | ${100}
        ${[200, 1000]}          | ${0}
        ${[1100, 1200]}         | ${100}
        ${[800, 1200, 2000]}    | ${67}
        ${[900, 1200]}          | ${50}
        ${[10, 800, 1200]}      | ${33}
        ${[0, 900, 1000, 1200]} | ${25}
    `(
        'Should return $expectedMetric as metric value for $recentTransferCosts transfer costs',
        async ({ recentTransferCosts, expectedMetric }) => {
            expect.assertions(2);
            const mock = mockFetchNetworkStatistics(Promise.resolve({ recentTransferCosts }));

            await getNetworkOverheadMetric().then(metric => {
                expect(mock).toHaveBeenCalledTimes(1);
                expect(metric).toEqual(expectedMetric);
            });
        }
    );

    it('Should return "-" is case of error', async () => {
        expect.assertions(1);
        mockFetchNetworkStatistics(Promise.reject());

        await getNetworkOverheadMetric().then(metric => {
            expect(metric).toEqual('-');
        });
    });
});

describe('fetchNetworkStatistics', () => {
    afterEach(() => restoreFetch);

    const fetchNetworkStatistics = Rewire.__GetDependency__('fetchNetworkStatistics');

    it('Should call proper URL', async () => {
        const fetchMock = mockFetchSuccess({ recentTransferCosts: [] });

        await fetchNetworkStatistics();

        expect(fetchMock)
            .toHaveBeenCalledTimes(1)
            .toBeCalledWith('/context/rest/static-asset-caching/network-statistics');
    });

    test.each`
        serverResponse                          | desc
        ${{ recentTransferCosts: [] }}          | ${'#1'}
        ${{ recentTransferCosts: [0] }}         | ${'#2'}
        ${{ recentTransferCosts: [1000] }}      | ${'#3'}
        ${{ recentTransferCosts: [100, 2000] }} | ${'#4'}
    `('Should resolve with network statistics on success ($desc)', async ({ serverResponse }) => {
        expect.assertions(2);
        const mock = mockFetchSuccess(serverResponse);

        await fetchNetworkStatistics().then(networkStats => {
            expect(mock).toHaveBeenCalledTimes(1);
            expect(networkStats).toEqual(serverResponse);
        });
    });

    test.each`
        serverResponse
        ${{ recentTransferCosts: null }}
        ${{ recentTransferCosts: undefined }}
        ${{ recentTransferCosts: {} }}
        ${{ recentTransferCosts: false }}
        ${{ recentTransferCosts: true }}
        ${{ recentTransferCosts: '' }}
        ${{ recentTransferCosts: 1000 }}
        ${null}
        ${{}}
        ${false}
        ${''}
    `(
        'Should reject with error when server responded with malformed data: $serverResponse',
        async ({ serverResponse }) => {
            expect.assertions(2);
            mockFetchSuccess(serverResponse);

            await fetchNetworkStatistics().catch(err => {
                expect(err).toBeInstanceOf(Error);
                expect(err.message).toEqual('Malformed network statistics received');
            });
        }
    );

    it('Should reject with error on server failure', async () => {
        mockFetchSuccess(null, false);

        await fetchNetworkStatistics().catch(err => {
            expect(err).toBeInstanceOf(Error);
            expect(err.message).toEqual('Fetch failed');
        });
    });

    it('Should reject on fetch failure', async () => {
        mockFetchFailure(new TypeError('Failed to fetch'));

        await fetchNetworkStatistics().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            expect(err.message).toEqual('Failed to fetch');
        });
    });
});
