import {
    performConnectivityChecksForTheInstance,
    performConnectivityChecksForCDN,
    __RewireAPI__ as HealthChecksRewire,
} from 'health-checks/health-checks';
import PerfAPI from 'shims/performance-api';
import ConnectivityResults from 'health-checks/connectivity-results';

jest.mock('shims/performance-api', () => ({
    __esModule: true,
    default: {},
}));

describe('performConnectivityChecksFor: the instance and CDN', () => {
    function mockPerformConnectivityChecks(returnValue) {
        const mock = jest.fn(() => returnValue);
        HealthChecksRewire.__Rewire__('performConnectivityChecks', mock);
        return mock;
    }

    let mock;
    beforeEach(() => {
        mock = mockPerformConnectivityChecks(
            new Promise(resolve => {
                process.nextTick(() => resolve(new ConnectivityResults()));
            })
        );
    });
    afterEach(() => {
        HealthChecksRewire.__ResetDependency__('performConnectivityChecks');
    });

    it('Verify ForTheInstance call', done => {
        const getBaseUrl = HealthChecksRewire.__GetDependency__('getBaseUrl');

        performConnectivityChecksForTheInstance().then(result => {
            expect(mock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(getBaseUrl(), true);
            expect(result).toBeInstanceOf(ConnectivityResults);
            done();
        });
    });

    it('Verify CDN call', done => {
        const cdnUrl = 'https://some.example.cdn.url';

        performConnectivityChecksForCDN(cdnUrl).then(result => {
            expect(mock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(`${cdnUrl}/context`);
            expect(result).toBeInstanceOf(ConnectivityResults);
            done();
        });
    });
});

describe('performConnectivityChecks', () => {
    function mockFetchTestResources(returnValue) {
        const mock = jest.fn(() => returnValue);
        HealthChecksRewire.__Rewire__('fetchTestResources', mock);
        return mock;
    }

    function mockAreTestResourcesServedOverHttp2(returnValue) {
        const mock = jest.fn(() => returnValue);
        HealthChecksRewire.__Rewire__('areTestResourcesServedOverHttp2', mock);
        return mock;
    }

    const performConnectivityChecks = HealthChecksRewire.__GetDependency__(
        'performConnectivityChecks'
    );

    afterEach(() => {
        HealthChecksRewire.__ResetDependency__('areTestResourcesServedOverHttp2');
        HealthChecksRewire.__ResetDependency__('fetchTestResources');
    });

    it('Verify the result when fetch succeeded and http2 was used for resources and a document', done => {
        const baseUrl = 'https://some.base.url';
        const testResourceUrls = ['test1', 'test2'];
        const fetchTestResourcesMock = mockFetchTestResources(Promise.resolve(testResourceUrls));
        const resourcesServedOverHttp2Mock = mockAreTestResourcesServedOverHttp2(true);

        performConnectivityChecks(baseUrl, true).then(result => {
            const expectedResult = new ConnectivityResults();
            expectedResult.canServeAssets = true;
            expectedResult.isReachable = true;
            expectedResult.http2 = true;

            expect(fetchTestResourcesMock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(baseUrl);
            expect(resourcesServedOverHttp2Mock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(testResourceUrls, true);
            expect(result)
                .toBeInstanceOf(ConnectivityResults)
                .toEqual(expectedResult);
            done();
        });
    });

    it('Verify the result when fetch succeeded and http2 was not used for resources', done => {
        const baseUrl = 'https://some.base.url/context';
        const testResourceUrls = ['test-url'];
        const fetchTestResourcesMock = mockFetchTestResources(Promise.resolve(testResourceUrls));
        const resourcesServedOverHttp2Mock = mockAreTestResourcesServedOverHttp2(false);

        performConnectivityChecks(baseUrl).then(result => {
            const expectedResult = new ConnectivityResults();
            expectedResult.canServeAssets = true;
            expectedResult.isReachable = true;
            expectedResult.http2 = false;

            expect(fetchTestResourcesMock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(baseUrl);
            expect(resourcesServedOverHttp2Mock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(testResourceUrls, false);
            expect(result)
                .toBeInstanceOf(ConnectivityResults)
                .toEqual(expectedResult);
            done();
        });
    });

    it('Verify the result when fetch failed', done => {
        const baseUrl = 'https://some.base.url:9999/context';
        const testResourceUrls = ['test-url', 'test-url2', 'test-url3'];
        const fetchTestResourcesMock = mockFetchTestResources(Promise.reject(testResourceUrls));
        const resourcesServedOverHttp2Mock = mockAreTestResourcesServedOverHttp2();

        performConnectivityChecks(baseUrl).then(result => {
            const expectedResult = new ConnectivityResults();
            expectedResult.canServeAssets = false;
            expectedResult.isReachable = false;

            expect(fetchTestResourcesMock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(baseUrl);
            expect(resourcesServedOverHttp2Mock).toHaveBeenCalledTimes(0);
            expect(result)
                .toBeInstanceOf(ConnectivityResults)
                .toEqual(expectedResult);
            done();
        });
    });
});

describe('areTestResourcesServedOverHttp2', () => {
    function mockEntriesByType(mockFn) {
        const mock = jest.fn((...args) => mockFn(...args));
        PerfAPI.getEntriesByType = mock;
        return mock;
    }

    const areTestResourcesServedOverHttp2 = HealthChecksRewire.__GetDependency__(
        'areTestResourcesServedOverHttp2'
    );
    let entries;

    beforeEach(() => {
        entries = {
            resource: [],
            navigation: [],
        };
        mockEntriesByType(type => entries[type]);
    });

    it('Should return undefined when performance api is not available', () => {
        delete PerfAPI.getEntriesByType;
        expect(areTestResourcesServedOverHttp2()).toEqual(undefined);
    });

    it('Should return undefined when there is no resource entries', () => {
        expect(areTestResourcesServedOverHttp2()).toEqual(undefined);
    });

    it('Should return undefined when there is no `nextHopProtocol` in the resource entry', () => {
        entries.resource.push({ name: '' });
        expect(areTestResourcesServedOverHttp2()).toEqual(undefined);
    });

    it('Should return undefined when resource entries do not match given resources', () => {
        const url = 'test';
        entries.resource.push({ name: url, nextHopProtocol: 'h2' });

        expect(areTestResourcesServedOverHttp2(['different-name'])).toEqual(undefined);
    });

    it('Should return undefined when there is no navigation entry', () => {
        entries.resource.push({ name: '', nextHopProtocol: 'h2' });
        expect(areTestResourcesServedOverHttp2([], true)).toEqual(undefined);
    });

    it('Should return undefined when there is no navigation entry and some resources are given', () => {
        const url = 'test';
        entries.resource.push({ name: url, nextHopProtocol: 'h2' });

        expect(areTestResourcesServedOverHttp2([url], true)).toEqual(undefined);
    });

    it('Should return false when at least one resource entry was not sent over http2', () => {
        const url = 'test';
        entries.resource.push({ name: url, nextHopProtocol: 'h2' });
        const url2 = 'test2';
        entries.resource.push({ name: url2, nextHopProtocol: 'http/1.1' });
        const url3 = 'test3';
        entries.resource.push({ name: url3, nextHopProtocol: 'h2' });

        expect(areTestResourcesServedOverHttp2([url])).toEqual(true);
        expect(areTestResourcesServedOverHttp2([url, url2])).toEqual(false);
        expect(areTestResourcesServedOverHttp2([url, url2, url3])).toEqual(false);
    });

    it('Should return false when navigation was not sent over http2 regardless of resource entries', () => {
        entries.navigation.push({ nextHopProtocol: 'http/1.1' });
        const url = 'test';
        entries.resource.push({ name: url, nextHopProtocol: 'h2' });
        const url2 = 'test2';
        entries.resource.push({ name: url2, nextHopProtocol: 'h2' });

        expect(areTestResourcesServedOverHttp2([url], true)).toEqual(false);
        expect(areTestResourcesServedOverHttp2([url, url2], true)).toEqual(false);
    });

    it('Should return true when resource entries match given resources and http2 was used', () => {
        const url = 'test';
        entries.resource.push({ name: url, nextHopProtocol: 'h2' });
        const url2 = 'test2';
        entries.resource.push({ name: url2, nextHopProtocol: 'h2' });
        const url3 = 'test3';
        entries.resource.push({ name: url3, nextHopProtocol: 'h2' });

        expect(areTestResourcesServedOverHttp2([url])).toEqual(true);
        expect(areTestResourcesServedOverHttp2([url, url2])).toEqual(true);
        expect(areTestResourcesServedOverHttp2([url, url2, url3])).toEqual(true);
    });

    it('Should return true when resource entries match given resources and http2 was used also for navigation', () => {
        entries.navigation.push({ nextHopProtocol: 'h2' });
        const url = 'test';
        entries.resource.push({ name: url, nextHopProtocol: 'h2' });
        const url2 = 'test2';
        entries.resource.push({ name: url2, nextHopProtocol: 'h2' });
        const url3 = 'test3';
        entries.resource.push({ name: url3, nextHopProtocol: 'h2' });

        expect(areTestResourcesServedOverHttp2([url], true)).toEqual(true);
        expect(areTestResourcesServedOverHttp2([url, url2], true)).toEqual(true);
        expect(areTestResourcesServedOverHttp2([url, url2, url3], true)).toEqual(true);
    });
});

describe('fetchTestResources', () => {
    function mockFetchTestImage(returnValue) {
        const mock = jest.fn(() => returnValue);
        HealthChecksRewire.__Rewire__('fetchTestImage', mock);
        return mock;
    }

    const fetchTestResources = HealthChecksRewire.__GetDependency__('fetchTestResources');

    afterEach(() => {
        HealthChecksRewire.__ResetDependency__('fetchTestImage');
    });

    it('Should return an array of test resource urls on success', done => {
        const baseUrlParam = 'https://example.com/context';
        const testResourceUrl = 'https://example.com/context/s/img.png';
        const fetchTestImageMock = mockFetchTestImage(Promise.resolve(testResourceUrl));

        fetchTestResources(baseUrlParam).then(urls => {
            expect(fetchTestImageMock)
                .toHaveBeenCalledTimes(1)
                .toBeCalledWith(baseUrlParam);
            expect(urls).toIncludeSameMembers([testResourceUrl]);
            done();
        });
    });

    it('Should reject a promise on failure', done => {
        const fetchTestImageMock = mockFetchTestImage(Promise.reject(new Error('')));

        fetchTestResources().catch(err => {
            expect(fetchTestImageMock).toHaveBeenCalledTimes(1);
            expect(err).not.toBeEmpty();
            done();
        });
    });
});

describe('fetchTestImage', () => {
    const { createElement } = document;
    const fetchTestImage = HealthChecksRewire.__GetDependency__('fetchTestImage');
    let imgStub = {};

    beforeAll(() => {
        Object.defineProperty(document, 'createElement', {
            value: jest.fn(() => imgStub),
        });
    });
    beforeEach(() => {
        imgStub = {};
    });

    afterAll(() => {
        document.createElement = createElement;
    });

    function getTestImageUrl(baseUrl = '', nocacheParam = '') {
        return `${baseUrl}/s/NOCACHE/_/download/resources/com.atlassian.plugins.static-assets-url:health-checks-test/health-checks/test-img.png?_=${nocacheParam}`;
    }

    function mockNocacheParam(nocacheValue = '') {
        jest.spyOn(Date.prototype, 'getTime').mockImplementation(() => nocacheValue);
    }

    it('Sets correct test image URL', done => {
        const nocache = 12345678;
        const baseUrl = 'https://testBaseUrl.com:7777/product';
        mockNocacheParam(nocache);

        fetchTestImage(baseUrl);

        process.nextTick(() => {
            expect(imgStub.src).toEqual(getTestImageUrl(baseUrl, nocache));
            done();
        });
    });

    it('Resolves a promise on success with the image URL', done => {
        mockNocacheParam();

        fetchTestImage('').then(src => {
            expect(src).toEqual(getTestImageUrl());
            done();
        });

        process.nextTick(() => {
            imgStub.onload();
        });
    });

    it('Rejects a promise on failure with error', done => {
        mockNocacheParam();

        fetchTestImage('').catch(err => {
            expect(err).toBeInstanceOf(Error);
            expect(err.message).toEqual('Downloading the image failed');
            done();
        });

        process.nextTick(() => {
            imgStub.onerror();
        });
    });
});

describe('getBaseUrl', () => {
    const getBaseUrl = HealthChecksRewire.__GetDependency__('getBaseUrl');

    it('Returns correct base url', () => {
        expect(getBaseUrl()).toEqual('http://localhost/context');
    });
});
