package com.atlassian.plugins.rest.module;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.osgi.factory.OsgiPlugin;
import com.atlassian.plugin.servlet.ServletModuleManager;
import com.atlassian.plugin.servlet.descriptors.ServletFilterModuleDescriptor;
import com.atlassian.plugin.util.validation.ValidationPattern;
import com.atlassian.plugins.rest.doclet.generators.AtlassianWadlGeneratorConfig;
import org.dom4j.Element;

import javax.servlet.Filter;

import static com.atlassian.plugins.rest.doclet.generators.AtlassianWadlGeneratorConfig.APPLICATION_XML;
import static com.atlassian.plugins.rest.doclet.generators.AtlassianWadlGeneratorConfig.GRAMMARS_XML;
import static com.atlassian.plugins.rest.doclet.generators.AtlassianWadlGeneratorConfig.RESOURCE_XML;
import static java.util.Objects.requireNonNull;

/**
 * The module descriptor for the REST servlet. Registered dynamically by the {@link RestModuleDescriptor}.
 * Uses the specific {@link RestDelegatingServletFilter}.
 */
public class RestServletFilterModuleDescriptor extends ServletFilterModuleDescriptor {
    private final RestDelegatingServletFilter restDelegatingServletFilter;
    private final RestApiContext restApiContext;
    private final static String DISABLE_WADL_PROPERTY = "com.sun.jersey.config.feature.DisableWADL";

    RestServletFilterModuleDescriptor(OsgiPlugin plugin, ModuleFactory moduleFactory, ServletModuleManager servletModuleManager, RestApiContext restApiContext) {
        super(requireNonNull(moduleFactory, "moduleFactory can't be null"), requireNonNull(servletModuleManager, "servletModuleManager can't be nul"));
        this.restApiContext = requireNonNull(restApiContext, "restApiContext can't be null");
        this.restDelegatingServletFilter = new RestDelegatingServletFilter(plugin, restApiContext);
    }

    @Override
    public void init(Plugin plugin, Element element) throws PluginParseException {
        super.init(plugin, element);

        //JRADEV-23134 : alllow OD to switch off WADL generation, via system property
        getInitParams().put(DISABLE_WADL_PROPERTY, System.getProperty(DISABLE_WADL_PROPERTY, "false"));

        //see if we have a resourcedoc.xml and if so, use the extended WADL generator
        if (resourcesAvailable(plugin, APPLICATION_XML, GRAMMARS_XML, RESOURCE_XML)) {
            getInitParams().put("com.sun.jersey.config.property.WadlGeneratorConfig", AtlassianWadlGeneratorConfig.class.getName());
        }
    }

    @Override
    protected void provideValidationRules(ValidationPattern pattern) {
    }

    @Override
    public String getName() {
        return "Rest Servlet Filter";
    }

    @Override
    public Filter getModule() {
        return restDelegatingServletFilter;
    }

    public String getBasePath() {
        return restApiContext.getApiPath();
    }

    public ApiVersion getVersion() {
        return restApiContext.getVersion();
    }

    private static boolean resourcesAvailable(Plugin plugin, String... resources) {
        for (final String resource : resources) {
            if (plugin.getResource(resource) == null) {
                return false;
            }
        }
        return true;
    }
}
