package com.atlassian.plugins.rest.module;

import java.util.Collections;
import java.util.Objects;
import java.util.Set;

import com.atlassian.plugin.module.ContainerManagedPlugin;
import com.google.common.collect.ImmutableSet;
import com.sun.jersey.api.core.ResourceConfig;
import com.sun.jersey.core.spi.component.ComponentContext;
import com.sun.jersey.core.spi.component.ComponentScope;
import com.sun.jersey.core.spi.component.ioc.IoCComponentProvider;
import com.sun.jersey.core.spi.component.ioc.IoCComponentProviderFactory;
import com.sun.jersey.core.spi.component.ioc.IoCManagedComponentProvider;
import com.sun.jersey.server.impl.container.servlet.JSPTemplateProcessor;
import com.sun.jersey.spi.resource.PerRequest;

import static com.google.common.collect.Sets.difference;
import static java.util.Collections.singleton;
import static java.util.Objects.requireNonNull;

/**
 * The OSGi component provider factory.
 * <p>
 * This will support <em>user defined</em> resources and providers only.
 */
public class OsgiComponentProviderFactory implements IoCComponentProviderFactory {
    // Classes that are added to the resource config that we can't autowire.
    private static final Set<Class<?>> EXCLUDE = singleton(JSPTemplateProcessor.class);

    /**
     * The plugin that this factory belongs to.
     */
    private final ContainerManagedPlugin plugin;

    /**
     * The set of <em>user defined</em> resources and providers.
     */
    private final Set<Class<?>> classes;

    private final Set<?> instances;

    public OsgiComponentProviderFactory(ResourceConfig resourceConfig, ContainerManagedPlugin plugin) {
        this.plugin = requireNonNull(plugin);

        // get the "user defined" resource and provider classes
        final Set<Class<?>> classes = requireNonNull(resourceConfig).getClasses();
        if (classes != null) {
            this.classes = difference(ImmutableSet.copyOf(classes), EXCLUDE);
        } else {
            this.classes = Collections.emptySet();
        }

        if (resourceConfig instanceof OsgiResourceConfig) {
            instances = ((OsgiResourceConfig) resourceConfig).getInstances();
        } else {
            instances = Collections.emptySet();
        }
    }

    public IoCComponentProvider getComponentProvider(final Class<?> c) {
        return getComponentProvider(null, c);
    }

    public IoCComponentProvider getComponentProvider(ComponentContext cc, Class<?> c) {
        if (!classes.contains(c)) {
            return null;
        }
        final Object instance = getInstance(c);
        return instance == null ? new ContainerManagedComponentProvider(plugin, c) : new InstanceOsgiComponentProvider(instance);
    }

    private Object getInstance(Class<?> c) {
        for (Object o : instances) {
            if (o.getClass().equals(c)) {
                return o;
            }
        }
        return null;
    }

    private static class ContainerManagedComponentProvider implements IoCManagedComponentProvider {
        private final ContainerManagedPlugin plugin;
        private final Class<?> componentClass;

        public ContainerManagedComponentProvider(ContainerManagedPlugin plugin, Class<?> componentClass) {
            this.plugin = plugin;
            this.componentClass = componentClass;
        }

        public Object getInstance() {
            return plugin.getContainerAccessor().createBean(componentClass);
        }

        public ComponentScope getScope() {
            // If the class is annotated with PerRequest, then it should return per request, otherwise,
            // default to singleton
            if (componentClass.getAnnotation(PerRequest.class) != null) {
                return ComponentScope.PerRequest;
            }
            return ComponentScope.Singleton;
        }

        public Object getInjectableInstance(Object o) {
            plugin.getContainerAccessor().injectBean(o);
            return o;
        }
    }

    private static class InstanceOsgiComponentProvider implements IoCManagedComponentProvider {
        private final Object instance;

        public InstanceOsgiComponentProvider(Object instance) {
            this.instance = requireNonNull(instance);
        }

        public ComponentScope getScope() {
            return ComponentScope.Singleton;
        }

        public Object getInstance() {
            return instance;
        }

        public Object getInjectableInstance(Object o) {
            return o;
        }
    }
}
