package com.atlassian.plugins.utils;


import java.io.ObjectInputFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AbstractPointcutAdvisor;
import org.springframework.context.support.AbstractApplicationContext;

import java.util.HashSet;
import java.util.Set;

/**
 * This is a global deserialization filter for Jira.
 * The root reason for adding the filter was the vulnerability to JMX RCE attacks.
 * It maintains a blocklist of some well known vulnerable classes that will be rejected when loading.
 * The blocklist classes is placed in deserialization-blocklist.properties file and copied from
 * <a href="https://github.com/FasterXML/jackson-databind/blob/jackson-databind-2.14.2/src/main/java/com/fasterxml/jackson/databind/jsontype/impl/SubTypeValidator.java">here</a>
 *
 * @since v9.8
 */
public class BlocklistDeserializationFilter implements ObjectInputFilter {
    private static final Logger log = LoggerFactory.getLogger(BlocklistDeserializationFilter.class);

    private final Set<String> blockedClasses;

    public BlocklistDeserializationFilter(Set<String> blockedClasses) {
        this.blockedClasses = new HashSet<>(blockedClasses);
    }

    @Override
    public Status checkInput(final FilterInfo filterInfo) {
        Class<?> rawClass = filterInfo.serialClass();
        if (rawClass == null) {
            return Status.ALLOWED;
        }
        if (rawClass.isArray()) {
            rawClass = getClassTypeOfArray(rawClass);
        }
        String className = rawClass.getName();
        Status classNameCheck = checkClassName(className);

        if (classNameCheck == Status.REJECTED) {
            return classNameCheck;
        } else {
            return checkForBelongingToSpecificClasses(rawClass);
        }
    }

    private Class<?> getClassTypeOfArray(Class<?> rawClass) {
        while (rawClass.isArray()) {
            rawClass = rawClass.getComponentType();
        }
        return rawClass;
    }

    private Status checkClassName(final String className) {
        if (blockedClasses.contains(className)) {
            log.debug("Deserialize step prevented for class: {}. This class is blocked", className);
            return Status.REJECTED;
        }
        return Status.ALLOWED;
    }

    private Status checkForBelongingToSpecificClasses(Class<?> rawClass) {
        if (rawClass.isInterface()) {
            return Status.ALLOWED;
        }
        String className = rawClass.getName();
        if (AbstractPointcutAdvisor.class.isAssignableFrom(rawClass) || AbstractApplicationContext.class.isAssignableFrom(rawClass)) {
            log.debug("Deserialize step prevented for class: {}.", className);
            return Status.REJECTED;
        }
        if (className.startsWith("com.mchange.v2.c3p0.") && className.endsWith("DataSource")) {
            log.debug("Deserialize step prevented for class: {}.", className);
            return Status.REJECTED;
        }
        return Status.ALLOWED;
    }

}
