package com.atlassian.plugins.service.plugin;

import com.atlassian.plugins.domain.model.plugin.PluginCompatibilityStatus;
import com.atlassian.plugins.domain.model.plugin.PluginVersion;
import com.atlassian.plugins.service.RestService;

import java.util.List;
import java.util.Map;

public interface PluginVersionService extends RestService<PluginVersion>
{
    public static final String PATH = "/pluginversion";

    public static final String PATH_FINDCOMPATIBLE = "/find/compatible";
    public static final String PATH_FINDUPDATES = "/find/updates";
    public static final String PATH_FINDFEATURED = "/find/featured";
    public static final String PATH_FINDCOMPATIBLEKEY = "/find/compatiblekey";
    public static final String PATH_FINDCOMPATIBILITYSTATUS = "/find/compatibilitystatus";
    public static final String PATH_FINDPOPULARPLUGINS = "/find/popular";
    public static final String PATH_FINDSUPPORTEDPLUGINS = "/find/supported";
    
    public static final String PARAM_QUERY = "q";
    public static final String PARAM_PLUGINKEYS = "pk";
    public static final String PARAM_PLUGINVERSIONS = "pv";

    /**
     * Finds the latest plugin version for the product and build number given, for each plugin whose title
     * and description contains any of the words in the query string.  If no compatible version is found for a
     * particular plugin, then no plugin version will be returned.
     * <p/>
     * The results should be sorted by relevance to the query string.
     *
     * @param productId The ID of the product
     * @param buildNumber The build number of the product to find the compatible version of
     * @param query The query to use to search plugins
     * @param max The maximum number of results to return, null if all results should be returned
     * @param offset The offset of the results to return, null to start at the beginning of the list
     * @param expand The list of association property names to expand
     * @return The list of compatible plugin versions for product id and build number
     */
    List<PluginVersion> findCompatiblePluginVersions(String productId, Long buildNumber, String query,
            Integer max, Integer offset, List<String> expand);
    // Post summit TODO: Add a sort parameter, that sorts by the following: relevance, category name then relevance,
    // license then relevance, vendor name then relevance, latest released, average rating


    /**
     * Finds the count of the compatible plugins for the product and build number given, for each plugin whose title
     * and description contains any of the words in the query string.
     *
     * @param productId The ID of the product
     * @param buildNumber The build number of the product to find the compatible version of
     * @param query The query to use to search plugins
     * @return The number of compatible plugin for product id and build number
     */
    // Long findCompatiblePluginVersionsCount(String productId, Long buildNumber, String query);
    // Post summit TODO: Implement this

    /**
     * Finds latest plugin version for the product and build number given,
     * for each plugin key.
     * <p/>
     * The results is set of latest plugin versions.
     *
     * @param productId The ID of the product
     * @param buildNumber The build number of the product to find the compatible version of
     * @param pluginKeysAndVersions set of plugin keys with the version to consider the "currently installed" version
     * @param max The maximum number of results to return, null if all results should be returned
     * @param offset The offset of the results to return, null to start at the beginning of the list
     * @param expand The list of association property names to expand
     * @return The set containing available updates
     */
    List<PluginVersion> findUpdates(String productId, Long buildNumber, Map<String, String> pluginKeysAndVersions,
        Integer max, Integer offset, List<String> expand);

    /**
     * Returns set of featured plugins compatible with the product and build number given.
     * Usually three or four plugins.
     * <p/>
     * The results is set of featured plugin versions.
     *
     * @param productId The ID of the product
     * @param buildNumber The build number of the product to find the compatible version of
     * @param max The maximum number of results to return, null if all results should be returned
     * @param offset The offset of the results to return, null to start at the beginning of the list
     * @param expand The list of association property names to expand
     * @return The set of featured plugins
     */
    List<PluginVersion> findFeaturedPlugins(String productId, Long buildNumber, Integer max, Integer offset, List<String> expand);

    /**
     * Finds all the compatible plugin version for the product and build number given, for the plugin of the given key.
     * If no compatible version is found for the given plugin, then no plugin version will be returned.
     * <p/>
     * The results should be sorted by version number asc.
     *
     * @param productId The ID of the product
     * @param buildNumber The build number of the product to find the compatible version of
     * @param pluginKey the key of the plugin
     * @param max The maximum number of results to return, null if all results should be returned
     * @param offset The offset of the results to return, null to start at the beginning of the list
     * @param expand The list of association property names to expand
     * @return The list of all the compatible plugin versions of the plugin for product id and build number, sorted by
     * 			version number asc. An empty list is returned if no compatible version is found
     */
    List<PluginVersion> findCompatiblePluginVersionsByPluginKey(String productId, Long buildNumber, String pluginKey,
        Integer max, Integer offset, List<String> expand);

    /**
     * Used to get information about whether or not it is safe to upgrade the plugin and the product.

     * @param productKey for example "confluence"
     * @param currentProductBuildNumber the current build number of the product
     * @param targetProductBuildNumber the build number of the product against which to compare plugin versions
     * @param pluginKeysAndVersions the plugins to retrieve compatibility status for. The map key should be the plugin
     * key, and the map value should be the version of the plugin
     * @param max The maximum number of results to return, null if all results should be returned
     * @param offset The offset of the results to return, null to start at the beginning of the list
     * @param expand The list of association property names to expand
     * @return List of compatibility statuses
     */
    List<PluginCompatibilityStatus> getCompatibilityStatus(String productKey, Long currentProductBuildNumber, Long targetProductBuildNumber,
        Map<String, String> pluginKeysAndVersions, Integer max, Integer offset, List<String> expand);

    /**
     * Finds the popular plugin version for the product and build number given.  If no compatible version is
     * found for a particular plugin, then no plugin version will be returned.
     *
     * @param productId The ID of the product
     * @param buildNumber The build number of the product to find the compatible version of
     * @param max The maximum number of results to return, null if all results should be returned
     * @param offset The offset of the results to return, null to start at the beginning of the list
     * @param expand The list of association property names to expand
     * @return The list of popular compatible plugin versions for product id and build number
     */
    List<PluginVersion> findPopularPluginVersions(String productId, Long buildNumber,
            Integer max, Integer offset, List<String> expand);

    /**
     * Finds the supported plugin version for the product and build number given.  If no compatible version is
     * found for a particular plugin, then no plugin version will be returned.
     *
     * @param productId The ID of the product
     * @param buildNumber The build number of the product to find the compatible version of
     * @param max The maximum number of results to return, null if all results should be returned
     * @param offset The offset of the results to return, null to start at the beginning of the list
     * @param expand The list of association property names to expand
     * @return The list of supported compatible plugin versions for product id and build number
     */
    List<PluginVersion> findSupportedPluginVersions(String productId, Long buildNumber,
            Integer max, Integer offset, List<String> expand);
}
