package com.atlassian.plugins.domain.search;

import com.atlassian.plugins.domain.model.category.CategoryType;
import com.atlassian.plugins.domain.model.plugin.PluginStatus;
import com.atlassian.plugins.domain.model.plugin.PluginVersionStatus;
import com.atlassian.plugins.domain.model.review.ReviewStatus;

import javax.xml.bind.annotation.*;
import java.util.*;


/**
 * Used to build dynamic search criteria
 */
@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
public class SearchCriteria {

    public static final String ENTITY_PREFIX = "x";

    public static final String COLLECTION = "[]";
    private static final String COLLECTION_PREFIX = "c";

    private Integer maxResults;
    private Integer offset;

    private boolean distinct = false;

    @XmlElement
    private final List<SearchExpression> expressions = new LinkedList<SearchExpression>();
    private String[] orderBy;

    protected static final String AND = "AND";
    protected static final String OR = "OR";

    public SearchCriteria() {
        this(null, null);
    }

    public SearchCriteria(Integer maxResults, Integer offest) {
        super();
        this.maxResults = maxResults;
        this.offset = offest;
    }

    public String toString() {
        return selectString("<ENTITY>", true);
    }

    /**
     * The values to be used in the actual query, excluding null ones, including ones from children criteria
     *
     * @return the query values, in order
     */
    @SuppressWarnings("unchecked")
    public List<Object> getValues() {
        List<Object> list = new LinkedList<Object>();
        for (SearchExpression ex : expressions) {
            if (ex.getCriteria() != null) {
                list.addAll(ex.getCriteria().getValues());
            } else if (ex.getOperation().isSubstituted()) {
                if (ex.getValue() instanceof Collection) {
                    list.addAll((Collection)ex.getValue());
                } else {
                    list.add(ex.getValue());
                }
            }
        }
        return list;
    }

    /**
     * Return the criteria as a String that can be used for counting the results of a search
     *
     * @param entity -- the name of the entity you are querying
     * @return HQL for counting
     */
    public String countString(String entity) {
        return doString(entity, true, true, 1);
    }

    /**
     * Return the criteria as a String that can be used for searching.
     *
     * @param entity -- the name of the entity you are querying
     * @return HQL for selecting
     */
    public String selectString(String entity) {
        return doString(entity, true, false, 1);
    }

    /**
     * Return the criteria as a String that can be used for searching.
     *
     * @param entity -- the name of the entity you are querying
     * @param doPrefix -- true if you want SELECT x FROM entity x included in the sql
     * @return HQL for selecting
     */
    public String selectString(String entity, boolean doPrefix) {
        return doString(entity, doPrefix, false, 1) ;
    }

    private String doString(String entity, boolean doPrefix, boolean count, int collection) {
        StringBuffer buffer = new StringBuffer();

        StringBuffer joinBuffer = new StringBuffer();

        for (SearchExpression expression : expressions) {

            //add the AND or OR if we've already added some expressions to the query
            if (buffer.length() > 0) {
                buffer.append(" ");
                buffer.append(expression.getType());
                buffer.append(" ");
            }

            //add either brackets or the operation
            if (expression.getCriteria() != null) {

                //this whole join thing is totally dodgy as we have to communicate the JOIN up to the parent somehow
                //atm, it assumes one join per query and probably only works when the joined sc is one level down
                String subcriteria = expression.getCriteria().doString(entity, false, false, ++collection);
                if (subcriteria.contains("LEFT OUTER JOIN")) {
                    String[] joinstring = subcriteria.split("LEFT OUTER JOIN");
                    joinBuffer.append(" LEFT OUTER JOIN");
                    joinBuffer.append(joinstring[1]);
                    subcriteria = joinstring[0];
                }

                buffer.append("(");
                buffer.append(subcriteria);
                buffer.append(")");

                //bring up any distinct flags
                if (expression.getCriteria().isDistinct() && !distinct) {
                    distinct = true;
                }

            } else {

                //handle a collection in the dotted fieldname
                if (expression.getFieldname().contains(COLLECTION)) {

                    //http://www.hibernate.org/250.html#A34
                    //"from Entity e where e.collection.property" does not work in 3.2
                    //"from Entity e join e.collection c where c.property" used insted

                    String[] components = expression.getFieldname().split("\\[\\]", 2);
                    joinBuffer.append(" LEFT OUTER JOIN ");
                    joinBuffer.append(ENTITY_PREFIX);
                    joinBuffer.append(".");
                    joinBuffer.append(components[0]);
                    joinBuffer.append(" ");
                    joinBuffer.append(COLLECTION_PREFIX+collection);

                    if (expression.getFunction() == null) {
                        buffer.append(COLLECTION_PREFIX+collection);
                        buffer.append(components[1]);
                    } else {
                        buffer.append(expression.getFunction());
                        buffer.append("(");
                        buffer.append(COLLECTION_PREFIX+collection);
                        buffer.append(components[1]);
                        buffer.append(")");
                    }

                }

                //no collection - add the left half of the operation
                else if (expression.getFunction() != null) {
                    buffer.append(expression.getFunction());
                    buffer.append("(");
                    buffer.append(ENTITY_PREFIX);
                    buffer.append(".");
                    buffer.append(expression.getFieldname());
                    buffer.append(")");
                } else {
                    buffer.append(ENTITY_PREFIX);
                    buffer.append(".");
                    buffer.append(expression.getFieldname());
                }

                //add a space and the operation
                buffer.append(" ");
                buffer.append(expression.getOperation().getOperation());

                //append a ? when the operation is substituted
                if (expression.getOperation().isSubstituted()) {
                    //if we have a collection
                    if (expression.getValue() instanceof Collection) {
                        Collection coll = (Collection)expression.getValue();
                        if (coll.size() > 0) {
                            buffer.append(" (");
                            for (int i = 0; i < coll.size(); i++) {
                                if (i > 0) buffer.append(", ");
                                buffer.append("?");
                            }
                            buffer.append(")");
                        }

                    } else {
                        buffer.append(" ?");
                    }
                }
            }
        }

        StringBuffer prefix = new StringBuffer();
        StringBuffer suffix = new StringBuffer();

        if (doPrefix) {

            boolean doDistinct = distinct;

            if (count) {
                if (doDistinct) {
                    prefix.append("SELECT count(DISTINCT ");
                    prefix.append(ENTITY_PREFIX);
                    prefix.append(".id) FROM ");
                } else {
                    prefix.append("SELECT count(*) FROM ");
                }
            } else {

                //the distinct stuff was added so that the funky joins sometimes created due to authorisation would not affect the number of rows returned
                //however, this stuffs things up when ordering by dotted fields -- you get sql errors such as ERROR: for SELECT DISTINCT, ORDER BY expressions must appear in select list
                //so, only do distinct when order by contains no dotted fields
                //this means that we will sometimes return the same object multiple times :(
                if (orderBy != null) {
                    for (String field : orderBy) {
                        if (field.contains(".")) {
                            doDistinct = false;
                            break;
                        }
                    }
                }

                prefix.append("SELECT ");
                if (doDistinct) prefix.append("DISTINCT ");
                prefix.append(ENTITY_PREFIX);
                prefix.append(" FROM ");

                if (orderBy != null && orderBy.length > 0) {
                    for (String field : orderBy) {
                        if (suffix.length() > 0) suffix.append(", ");
                        suffix.append(ENTITY_PREFIX);
                        suffix.append(".");
                        suffix.append(field);
                    }
                    suffix.insert(0, " ORDER BY ");
                }
            }
            prefix.append(entity);
            prefix.append(" ");
            prefix.append(ENTITY_PREFIX);
            if (joinBuffer.length() > 0) prefix.append(joinBuffer);
            if (!expressions.isEmpty()) prefix.append(" WHERE ");

        } else {
            if (joinBuffer.length() > 0) {
                suffix.append(joinBuffer);
            }
        }

        return prefix.toString() + buffer.toString() + suffix.toString();
    }

    /**
     * Start a query
     *
     * @param fieldName the field name to select on, can be dotted (eg account.merchandise) - cannot be null
     * @param operation the search expression, such as equals or less than - cannot be null
     * @param value the value to search on, can be null
     * @param function the function to apply to the fieldName (eg lower(email)), can be null
     * @return itself
     */
    public SearchCriteria where(String fieldName, SearchOperation operation, Object value, String function) {
        return and(fieldName, operation, value, function);
    }

    /**
     * AND a criteria
     *
     * @param fieldName the field name to select on, can be dotted (eg account.merchandise) - cannot be null
     * @param operation the search expression, such as equals or less than - cannot be null
     * @param value the value to search on, can be null
     * @param function the function to apply to the fieldName (eg lower(email)), can be null
     * @return itself
     */
    public SearchCriteria and(String fieldName, SearchOperation operation, Object value, String function) {
        expressions.add(new SearchExpression(AND, fieldName, operation, value, function));
        return this;
    }

    /**
     * OR a criteria
     *
     * @param fieldName the field name to select on, can be dotted (eg account.merchandise) - cannot be null
     * @param operation the search expression, such as equals or less than - cannot be null
     * @param value the value to search on, can be null
     * @param function the function to apply to the fieldName (eg lower(email)), can be null
     * @return itself
     */
    public SearchCriteria or(String fieldName, SearchOperation operation, Object value, String function) {
        expressions.add(new SearchExpression(OR, fieldName, operation, value, function));
        return this;
    }

    /**
     * Convenience method with a null function
     *
     * @param fieldName the field name to select on, can be dotted (eg account.merchandise) - cannot be null
     * @param operation the search expression, such as equals or less than - cannot be null
     * @param value the value to search on, can be null
     * @return itself
     */
    public SearchCriteria where(String fieldName, SearchOperation operation, Object value) {
        return where(fieldName, operation, value, null);
    }

    /**
     * Convenience method with a null function
     *
     * @param fieldName the field name to select on, can be dotted (eg account.merchandise) - cannot be null
     * @param operation the search expression, such as equals or less than - cannot be null
     * @param value the value to search on, can be null
     * @return itself
     */
    public SearchCriteria and(String fieldName, SearchOperation operation, Object value) {
        return and(fieldName, operation, value, null);
    }

    /**
     * Convenience method with a null function
     *
     * @param fieldName the field name to select on, can be dotted (eg account.merchandise) - cannot be null
     * @param operation the search expression, such as equals or less than - cannot be null
     * @param value the value to search on, can be null
     * @return itself
     */
    public SearchCriteria or(String fieldName, SearchOperation operation, Object value) {
        return or(fieldName, operation, value, null);
    }

    /**
     * Creates the specified search criteria in brackets. Extremely useful with ORs.
     *
     * e.g. with
     *
     * SearchCriteria criteria1 = new SearchCriteria
     *      .where("height", SearchOperation.GREATERTHAN, 10)
     *      .or("height", SearchOperation.LESSTHAN, 20);
     *
     * SearchCriteria criteria2 = new SearchCriteria().where(c)
     *
     * Upon rendering, criteria2 will give:
     *
     * where (height > 10 or height < 20)
     *
     * Whereas rendering criteria1 will give:
     *
     * where height > 10 or height < 20
     *
     * @param criteria - the criteria for inside the brackets
     * @return itself
     *
     */
    public SearchCriteria where(SearchCriteria criteria) {
        return and(criteria);
    }

    /**
     * Creates the specified search criteria in brackets
     *
     * @param criteria - the criteria for inside the brackets
     * @return itself
     */
    public SearchCriteria and(SearchCriteria criteria) {
        expressions.add(new SearchExpression(AND, criteria));
        return this;
    }

    /**
     * Creates the specified search criteria in brackets
     *
     * @param criteria - the criteria for inside the brackets
     * @return itself
     */
    public SearchCriteria or(SearchCriteria criteria) {
        expressions.add(new SearchExpression(OR, criteria));
        return this;
    }

    /**
     * Returns the results ordered by specified fields. Is ignored for count queries.
     *
     * @param fields - a list of fields, in order, to sort the results by. May include "ASC" or "DESC" in the field, e.g. name DESC
     * @return itself
     */
    public SearchCriteria orderBy(String... fields) {
        this.orderBy = fields;
        return this;
    }

    public List<String> getOrderBy() {
        if (orderBy == null) return null;
        return Collections.unmodifiableList(Arrays.asList(orderBy));
    }

    public List<SearchExpression> getExpressions() {
        return Collections.unmodifiableList(expressions);
    }

    public Integer getMaxResults() {
        return maxResults;
    }

    public Integer getOffset() {
        return offset;
    }

    public void setMaxResults(Integer maxResults) {
        this.maxResults = maxResults;
    }

    public void setOffset(Integer offset) {
        this.offset = offset;
    }

    public boolean isDistinct() {
        return distinct;
    }

    public void setDistinct(boolean distinct) {
        this.distinct = distinct;
    }

    @XmlRootElement
    @XmlAccessorType(XmlAccessType.FIELD)
    @XmlSeeAlso({CategoryType.class, PluginStatus.class, PluginVersionStatus.class, ReviewStatus.class})
    public static class SearchExpression {

        private String type;
        private String fieldname;
        private SearchOperation operation;
        private Object value;
        private String function;
        private SearchCriteria criteria;

        public SearchExpression(String type, SearchCriteria criteria) {
            if (criteria == null) throw new IllegalArgumentException("Cannot have a null criteria");
            this.type = type;
            this.criteria = criteria;
        }

        public SearchExpression(String type, String fieldname, SearchOperation operation, Object value, String function) {
            if (fieldname == null) throw new IllegalArgumentException("Cannot have a null field name");
            if (operation == null) throw new IllegalArgumentException("Cannot have a null search operation");
            if (operation.isSubstituted() && value == null) throw new IllegalArgumentException("Search operation "+ operation + " requires a value");
            if (!operation.isSubstituted() && value != null) throw new IllegalArgumentException("Search operation "+ operation + " does not require a value");
            this.type = type;
            this.fieldname = fieldname;
            this.operation = operation;
            this.value = value;
            this.function = function;
        }

        private SearchExpression() {
            //no args constructor needed for jaxb -- otherwise all the fields could be final
        }

        public String getType() {
            return type;
        }

        public String getFieldname() {
            return fieldname;
        }

        public SearchOperation getOperation() {
            return operation;
        }

        public Object getValue() {
            return value;
        }

        public String getFunction() {
            return function;
        }

        public SearchCriteria getCriteria() {
            return criteria;
        }
    }

}
