package com.atlassian.plugins.domain;

import com.atlassian.plugins.domain.model.Link;

import javax.xml.bind.Marshaller;
import javax.xml.bind.annotation.*;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

@XmlAccessorType(XmlAccessType.FIELD)
public abstract class AbstractDTO implements DTO {

    @XmlAttribute
    @XmlID
    private String id;

    @XmlAttribute
    private String expand;
    
    @XmlElement
    private Link link;

    /**
     * When the object was last updated - may be null
     */
    @XmlElement
    private Date lastModified;

    /**
     * Who updated the object last - may be null
     */
    @XmlElement
    private String lastModifiedBy;

    /**
     * Whether or not the object has had its fields retrieved (expanded) or just a references to itself
     */
    @XmlTransient
    private boolean isExpanded = false;

    public String toString() {
        return getClass().getName() + " [" + getId() + "]";
    }

    public int hashCode() {
        String id = getId();
        if (id == null) return super.hashCode();
        return id.hashCode();
    }

    public boolean equals(DTO other) {
        if (other == null) return false;
        String id = getId();
        String otherid = other.getId();
        if (id == null && otherid == null) return super.equals(other);
        if (id == null) return false;
        return id.equals(otherid);
    }

    /**
     * This method is called by JAXB before an object is marshalled.
     *
     * We use it to nullify all fields that we don't want marshalled when it's a "link only" element
     * (ie not expanded).
     *
     * It's abstract on this class because it can't be implemented up here and called when the child is marshalled,
     * but we want to make sure that it's on each of the DTOs so that we've at least thought about it when writing a
     * DTO.
     *
     * @param marshaller - the marshaller
     * @return true
     */
    public abstract boolean beforeMarshal(Marshaller marshaller);

    /**
     * This method is called by JAXB after an object is marshalled.
     *
     * We use it to de-nullify all fields that were affected by beforeMarshal().
     *
     * It's abstract on this class because it can't be implemented up here and called when the child is marshalled,
     * but we want to make sure that it's on each of the DTOs so that we've at least thought about it when writing a
     * DTO.
     *
     * @param marshaller - the marshaller
     */
    public abstract void afterMarshal(Marshaller marshaller);

    @XmlTransient
    private Map<String, Object> cache = new HashMap<String, Object>();

    protected void put(String key, Object obj) {
        cache.put(key, obj);
    }

    protected Object get(String key) {
        return cache.remove(key);
    }

    public boolean isExpanded() {
        return isExpanded;
    }

    public void setExpanded(boolean expanded) {
        isExpanded = expanded;
    }

    public String getLastModifiedBy() {
        return lastModifiedBy;
    }

    public void setLastModifiedBy(String lastModifiedBy) {
        this.lastModifiedBy = lastModifiedBy;
    }

    public Map<String, Object> getCache() {
        return cache;
    }

    public void setCache(Map<String, Object> cache) {
        this.cache = cache;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public Link getLink() {
        return link;
    }

    public void setLink(Link link) {
        this.link = link;
    }

    public String getExpand() {
        return expand;
    }

    public void setExpand(String expand) {
        this.expand = expand;
    }

    public Date getLastModified() {
        return lastModified;
    }

    public void setLastModified(Date lastModified) {
        this.lastModified = lastModified;
    }
}
