package com.atlassian.plugins.less;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import com.google.common.io.LineReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class UriDependencyCollector {

    private static final Pattern IMPORT_URI_PATTERN = Pattern.compile("" +
            "@import(?:-once)?\\s+" + // import statement
            "(?:\\(([^)]+)\\)\\s+)?" + // import options
            "\"([^\"]+)\";"); // import uri

    private static final int GROUP_IMPORT_OPTIONS = 1;
    private static final int GROUP_IMPORT_URI = 2;

    private static final Logger log = LoggerFactory.getLogger(UriDependencyCollector.class);

    private final UriResolverManager uriResolverManager;

    public UriDependencyCollector(UriResolverManager uriResolverManager) {
        this.uriResolverManager = uriResolverManager;
    }

    /**
     * extract the import dependencies for the supplied uri.
     * <p>
     * Its horrible that we have to do this. Unfortunately this is the only way to accurately determine
     * when a LESS resource changes
     */
    public Set<URI> getDependencies(final URI baseUri) {
        final Set<URI> collector = Sets.newLinkedHashSet(); // Use a linked hash set so that the iteration is stable
        try (Reader reader = new InputStreamReader(uriResolverManager.getResolverOrThrow(baseUri).open(baseUri))) {
            LineReader lineReader = new LineReader(reader);
            String line;
            while ((line = lineReader.readLine()) != null) {
                collectUris(collector, baseUri, line);
            }
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
        return Collections.unmodifiableSet(collector);
    }

    @VisibleForTesting
    void collectUris(Set<URI> collector, URI baseUri, String chunk) {
        Matcher matcher = IMPORT_URI_PATTERN.matcher(chunk);
        while (matcher.find()) {
            ImportOption importOption = ImportOption.fromString(matcher.group(GROUP_IMPORT_OPTIONS));
            String path = matcher.group(GROUP_IMPORT_URI);
            if (path != null) {
                path = importOption.amendURI(path);
                URI uri = baseUri.resolve(path);
                if (uriResolverManager.isUriSupported(uri)) {
                    collector.add(uri);
                } else {
                    log.warn("Ignoring LESS uri as it is not supported. uri={}", uri);
                }
            }
        }
    }

    private enum ImportOption {
        CSS,
        INLINE {
            @Override
            String amendURI(String uri) {
                if (!uri.endsWith(".css")) {
                    uri += ".css";
                }
                return uri;
            }
        },
        LESS,
        NONE {
            @Override
            String amendURI(String uri) {
                if (!uri.endsWith(".less")) {
                    uri += ".less";
                }
                return uri;
            }
        };

        String amendURI(String uri) {
            return uri;
        }

        private static ImportOption fromString(String value) {
            if (value != null) {
                value = value.toUpperCase(Locale.US);
                for (ImportOption importOption : values()) {
                    if (importOption.name().equals(value)) {
                        return importOption;
                    }
                }
            }
            return NONE;
        }
    }
}
