package com.atlassian.plugins.less;

import com.atlassian.lesscss.spi.UriResolver;
import com.atlassian.plugin.util.PluginUtils;
import com.google.common.annotations.VisibleForTesting;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

public class PreCompilationUtils {

    private static final String PROP_USE_PRE_COMPILE = "atlassian.lesscss.use.precompiled";

    private PreCompilationUtils() {
        throw new UnsupportedOperationException();
    }

    /**
     * Retrieve the URI of the corresponding pre-compiled URI, or {@code null} if one does not exist, or
     * pre-compilation is disabled.
     *
     * @param uriResolver the URI resolver which {@link UriResolver#supports(java.net.URI) supports} the URI
     * @param unCompiledUri the un-compiled URI location
     * @return the pre-compiled URI or {@code null}
     */
    public static URI resolvePreCompiledUri(UriResolver uriResolver, URI unCompiledUri) {
        // Short circuit if pre-compilation isn't enabled
        if (!isPreCompileEnabled()) {
            return null;
        }

        for (URI uri : getPrecompiledAlternatives(unCompiledUri)) {
            if (uriResolver.exists(uri)) {
                return uri;
            }
        }

        return null;
    }

    @VisibleForTesting
    static List<URI> getPrecompiledAlternatives(URI unCompiledUri) {
        String unCompiledLocation = unCompiledUri.getSchemeSpecificPart();
        int lastDot = unCompiledLocation.lastIndexOf('.');

        // Defensive coding. This should never happen but just in case
        if (lastDot == -1) {
            return Collections.emptyList();
        }

        // strip extension the extension
        String locationWithoutExtension = unCompiledLocation.substring(0, lastDot);

        List<URI> precompiledUris = new LinkedList<>();
        for (String extension : Arrays.asList(".less.css", "-less.css")) {
            try {
                String newSchemeSpecificPart = locationWithoutExtension + extension;
                URI uri = new URI(unCompiledUri.getScheme(), newSchemeSpecificPart, unCompiledUri.getFragment());
                precompiledUris.add(unCompiledUri.resolve(uri));
            } catch (URISyntaxException e) {
                throw new IllegalArgumentException(e);
            }
        }
        return precompiledUris;
    }

    private static boolean isPreCompileEnabled() {
        // If we're in dev mode and the pre-compilation flag isn't explicitly set to true
        // don't use the pre-compiled version. Alternatively if we are in production mode,
        // check if the flag is explicitly false
        String prop = System.getProperty(PROP_USE_PRE_COMPILE);
        Boolean usePreCompiled = prop != null ? Boolean.parseBoolean(prop) : null;
        return !(Boolean.getBoolean(PluginUtils.ATLASSIAN_DEV_MODE) && !Boolean.TRUE.equals(usePreCompiled)) || Boolean.FALSE.equals(usePreCompiled);
    }

}
