package com.atlassian.plugins.less;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.lesscss.Constants;
import com.atlassian.lesscss.spi.DimensionAwareUriResolver;
import com.atlassian.lesscss.spi.EncodeStateResult;
import com.atlassian.lesscss.spi.UriResolverStateChangedEvent;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.StateAware;
import com.atlassian.plugin.event.PluginEventListener;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.event.events.PluginDisabledEvent;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.webresource.api.prebake.Coordinate;
import com.atlassian.webresource.api.prebake.Dimensions;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.util.Optional;

import static com.google.common.base.Strings.isNullOrEmpty;

public class PluginUriResolver implements DimensionAwareUriResolver, StateAware {

    private static final String SNAPSHOT_VERSION = "SNAPSHOT";
    private final EventPublisher eventPublisher;
    private final PluginAccessor pluginAccessor;
    private final PluginEventManager pluginEventManager;

    public PluginUriResolver(EventPublisher eventPublisher,
                             PluginAccessor pluginAccessor,
                             PluginEventManager pluginEventManager) {
        this.eventPublisher = eventPublisher;
        this.pluginAccessor = pluginAccessor;
        this.pluginEventManager = pluginEventManager;
    }

    @Override
    public Dimensions computeDimensions() {
        return Dimensions.empty();
    }

    @Override
    public void enabled() {
        pluginEventManager.register(this);
    }

    @Override
    public void disabled() {
        pluginEventManager.unregister(this);
    }

    @Override
    public boolean exists(URI uri) {
        Plugin plugin = resolvePlugin(uri);
        String path = getResourcePath(uri);
        return plugin != null && plugin.getResource(path) != null;
    }

    @Override
    public EncodeStateResult encodeState(URI uri, Coordinate coord) {
        return new EncodeStateResult(encodeState(uri), Optional.empty());
    }

    @Override
    public String encodeState(URI uri) {
        Plugin plugin = resolvePlugin(uri);
        final String version = plugin.getPluginInformation().getVersion();

        // For snapshot versions we are encoding the resource with the last modified so the hashes will be less reluctant.
        // Unfortunately the last modified date is the date when the plugin was loaded into the system so in other cases
        // we need to encode based on plugin version to keep the hashes stable in production environment.
        if (version.endsWith(SNAPSHOT_VERSION)) {
            return encodeFromDateLastModified(uri, plugin);
        }

        return version;
    }

    private String encodeFromDateLastModified(URI uri, Plugin plugin)
    {
        URL url = plugin.getResource(getResourcePath(uri));

        URLConnection connection = null;
        try {
            connection = url.openConnection();
            return String.valueOf(connection.getLastModified());
        } catch (IOException e) {
            throw new IllegalStateException(e);
        } finally {
            if (connection != null) {
                try {
                    connection.getInputStream().close();
                } catch (IOException ignored) {
                }
            }
        }
    }

    @PluginEventListener
    public void onPluginDisabled(PluginDisabledEvent event) {
        eventPublisher.publish(new PluginUriResolvedStateChangedEvent(this, event.getPlugin().getKey()));
    }

    @PluginEventListener
    public void onPluginEnabled(PluginEnabledEvent event) {
        eventPublisher.publish(new PluginUriResolvedStateChangedEvent(this, event.getPlugin().getKey()));
    }

    @Override
    public InputStream open(URI uri) throws IOException {
        Plugin plugin = resolvePlugin(uri);

        InputStream in = plugin.getResourceAsStream(getResourcePath(uri));
        if (in == null) {
            throw new IOException(uri.getPath() + " does not exist in plugin " + plugin.getKey());
        }

        return in;
    }

    @Override
    public boolean supports(URI uri) {
        return Constants.SCHEME_PLUGIN.equals(uri.getScheme()) && !isNullOrEmpty(uri.getHost());
    }

    /**
     * Workaround for PLUG-1104.
     */
    private String getResourcePath(URI uri) {
        String path = uri.getPath();
        if (!isNullOrEmpty(path) && path.startsWith("/")) {
            path = path.substring(1);
        }
        return path;
    }

    private Plugin resolvePlugin(URI uri) {
        return pluginAccessor.getPlugin(uri.getHost());
    }

    public static class PluginUriResolvedStateChangedEvent extends UriResolverStateChangedEvent {

        private final String pluginKey;

        public PluginUriResolvedStateChangedEvent(Object source, String pluginKey) {
            super(source);
            this.pluginKey = pluginKey;
        }

        @Override
        public boolean hasChanged(URI uri) {
            return pluginKey.equals(uri.getHost());
        }
    }

}
