package com.atlassian.plugins.less;

import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.lesscss.spi.DimensionAwareUriResolver;
import com.atlassian.lesscss.spi.EncodeStateResult;
import com.atlassian.lesscss.spi.UriResolver;
import com.atlassian.lesscss.spi.UriResolverStateChangedEvent;
import com.atlassian.webresource.api.assembler.resource.PrebakeError;
import com.atlassian.webresource.api.prebake.Coordinate;
import com.google.common.base.Joiner;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

public class CachingUriStateManager implements UriStateManager {

    private static final Logger log = LoggerFactory.getLogger(CachingUriStateManager.class);

    private final LoadingCache<URI, UriInfo> cache;
    private final EventPublisher eventPublisher;
    private final UriResolverManager uriResolverManager;
    private final UriDependencyCollector uriDependencyCollector;

    public CachingUriStateManager(EventPublisher eventPublisher,
                                  UriResolverManager uriResolverManager,
                                  UriDependencyCollector uriDependencyCollector) {
        this.eventPublisher = eventPublisher;
        this.uriResolverManager = uriResolverManager;
        this.uriDependencyCollector = uriDependencyCollector;

        this.cache = CacheBuilder.newBuilder()
                .build(new CacheLoader<URI, UriInfo>() {
                    @Override
                    public UriInfo load(URI uri) throws Exception {
                        return computeUriInfo(uri);
                    }
                });
    }

    public void registerEventListeners() throws Exception {
        eventPublisher.register(this);
    }

    public void unRegisterEventListeners() throws Exception {
        eventPublisher.unregister(this);
    }

    @Override
    public String getState(URI uri) {
        List<String> states = Lists.newArrayList();
        collectUriState(Sets.newHashSet(uri), states, uri, true, cache::getUnchecked);

        return Joiner.on(',').join(states);
    }

    @Override
    public PrebakeStateResult getState(URI uri, Coordinate coord) {
        List<String> states = new ArrayList<>();
        List<PrebakeError> prebakeErrors = new ArrayList<>();
        collectUriState(Sets.newHashSet(uri), states, uri, true, _uri -> {
            PrebakeComputeUriInfoResult prebakeErrorsAndInfo = computeUriInfo(_uri, coord);
            prebakeErrorsAndInfo.prebakeError.ifPresent(prebakeErrors::add);
            return prebakeErrorsAndInfo.uriInfo;
        });

        return new PrebakeStateResult(Joiner.on(',').join(states), prebakeErrors);
    }

    @EventListener
    public void onStateChanged(UriResolverStateChangedEvent event) {
        for (Iterator<Map.Entry<URI, UriInfo>> it = cache.asMap().entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<URI, UriInfo> entry = it.next();
            if (event.hasChanged(entry.getKey())) {
                log.debug("LESS has changed. Expiring lastModified cache. uri={}", entry.getKey());
                it.remove();
            }
        }
    }

    private void collectUriState(Set<URI> alreadySeen,
                                 List<String> states,
                                 URI uri,
                                 boolean root,
                                 Function<URI, UriInfo> computeUriInfo) {
        UriInfo value = computeUriInfo.apply(uri);
        if (root && value.preCompiledState != null) {
            // If the pre-compiled version is being used, the dependencies will never be
            // used to produce the contents so we short-circuit the method. However we
            // Only do this for the 'root' URI. If we are traversing a dependencies state
            // we will be using the un-compiled version.
            states.add(value.preCompiledState);
            return;
        }

        states.add(value.state);

        for (URI dependency : value.dependencies) {
            if (alreadySeen.add(dependency)) {
                collectUriState(alreadySeen, states, dependency, false, computeUriInfo);
            }
        }
    }

    private UriInfo computeUriInfo(URI uri) {
        log.debug("Computing LESS uri info. uri={}", uri);
        UriResolver uriResolver = uriResolverManager.getResolverOrThrow(uri);
        URI preCompiledUri = PreCompilationUtils.resolvePreCompiledUri(uriResolver, uri);
        Set<URI> dependencies = uriDependencyCollector.getDependencies(uri);

        return new UriInfo(
                dependencies,
                preCompiledUri == null ? null : uriResolver.encodeState(preCompiledUri),
                uriResolver.encodeState(uri)
        );
    }

    private PrebakeComputeUriInfoResult computeUriInfo(URI uri, Coordinate coord) {
        log.debug("Computing LESS uri info. uri={}", uri);
        UriResolver uriResolver = uriResolverManager.getResolverOrThrow(uri);
        URI preCompiledUri = PreCompilationUtils.resolvePreCompiledUri(uriResolver, uri);
        Set<URI> dependencies = uriDependencyCollector.getDependencies(uri);

        if (preCompiledUri != null) {
            String precompiledState = uriResolver.encodeState(preCompiledUri);
            UriInfo uriInfo = new UriInfo(dependencies, precompiledState, uriResolver.encodeState(uri));

            return new PrebakeComputeUriInfoResult(uriInfo, Optional.empty());
        } else if (uriResolver instanceof DimensionAwareUriResolver) {
            DimensionAwareUriResolver dimensionAwareUriResolver = (DimensionAwareUriResolver) uriResolver;
            EncodeStateResult encodeStateResult = dimensionAwareUriResolver.encodeState(uri, coord);
            UriInfo uriInfo = new UriInfo(dependencies, null, encodeStateResult.getState());

            return new PrebakeComputeUriInfoResult(uriInfo, encodeStateResult.getPrebakeError());
        } else {
            String state = uriResolver.encodeState(uri);
            UriInfo uriInfo = new UriInfo(dependencies, null, state);
            PrebakeError prebakeError =
                    new DimensionUnawareUriResolverPrebakeError<>(uriResolver);

            return new PrebakeComputeUriInfoResult(uriInfo, Optional.of(prebakeError));
        }
    }

    private static class UriInfo {

        private final Set<URI> dependencies;
        private final String preCompiledState;
        private final String state;

        private UriInfo(Set<URI> dependencies, String preCompiledState, String state) {
            this.dependencies = dependencies;
            this.preCompiledState = preCompiledState;
            this.state = state;
        }
    }

    private static class PrebakeComputeUriInfoResult {

        private final UriInfo uriInfo;
        private final Optional<PrebakeError> prebakeError;

        public PrebakeComputeUriInfoResult(UriInfo uriInfo, Optional<PrebakeError> prebakeError) {
            this.uriInfo = uriInfo;
            this.prebakeError = prebakeError;
        }
    }


}
