package com.atlassian.plugins.avatar;

import java.io.IOException;
import java.io.InputStream;

/**
 * Representation of an Avatar image for a user (or possibly some other application specific entity). The URL may be
 * hosted on an external system such that some metadata about the Avatar are not known by the implementation, for
 * example whether it is the default avatar, provided by the system or user-customised.
 */
@SuppressWarnings("UnusedDeclaration")
public interface PluginAvatar {

    /**
     * String representation of the primary unique id of the user (or other entity) that the Avatar belongs to.
     *
     * @return the id.
     */
    public String getOwnerId();

    /**
     * Return a "URI reference" (a URL that may be relative to a known base or absolute) that can be used to fetch the
     * image data for this Avatar. The URL does not have to be internet-accessible, but if it is not, then
     * {@link #isExternal()} must return false. URLs to in-product-served Avatars are expected to operate on multiple
     * domains because sometimes domains and bases differ from that of the request. In such cases, it is important that
     * the URL be relative to a known base rather than constructing an absolute using the request or system config
     * to determine the base. Proxies and inter-server app links will not necessarily be using a client-accessible URL.
     * In other words, if an implementation supplies URLs for avatars served by an in-product servlet, they'd better be
     * relative. Absolute URLs are ideal for implementations that use services like Gravatar or Atlassian ID Avatars.
     *
     * @return the URL.
     */
    public String getUrl();

    /**
     * Actual pixel size.
     *
     * @return avatar size in pixels.
     */
    public int getSize();

    /**
     * The MIME type for the Avatar.
     *
     * @return image/png or whatever.
     */
    public String getContentType();

    /**
     * Whether the URL for the Avatar is hosted on an external and internet-accessible service like Gravatar, Atlassian
     * ID avatars or some other universally accessible service. If uncertain, return false. Note this also implies that
     * it would probably be silly to call {@link #getBytes()}.
     *
     * @return false unless the server behind this Avatar is guaranteed internet-accessible.
     */
    public boolean isExternal();

    /**
     * Provides an InputStream for reading the image data. Callers must close the stream. DO NOT call this method if the
     * URL is public unless you somehow know that the configuration is OK with this. External Avatars implies the bytes
     * could be read off the network which the caller can always do for themselves if they really need them.
     * Implementations may decide to do this but the caller should take responsibility. Alternately, implementations may
     * throw IOException to indicate the bytes are unavailable or contraband.
     *
     * @return a stream for reading the image data.
     * @throws IOException if called when returns true or if the bytes cannot be found.
     */
    public InputStream getBytes() throws IOException;

    /**
     * Provides this avatar in another size.
     *
     * @param size the required size of avatar
     * @return new instance of {@link com.atlassian.plugins.avatar.PluginAvatar}
     */
    public PluginAvatar atSize(int size);
}
