package com.atlassian.plugins.avatar;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Interface for services that provide avatar images for a users (or other domain objects). The typical use case for
 * avatars is users, although this interface doesn't care if they are users. T is the type of the domain object the
 * avatar is for and I is the type of the id of the Avatar objects (if any).
 */
@SuppressWarnings("UnusedDeclaration")
public interface AvatarProvider<T, I> {

    /**
     * Gets the Avatar for the given user at the given size. If the size is not possible, a best effort is made, to
     * return one with a size that is usable in a user-facing client. This should be the next larger size if one is
     * available, otherwise the next smaller size. If the size cannot be interpreted, the default size should be
     * returned.
     * <p>
     * Implementations that cannot provide an avatar for the given user can return a URL for default/unknown users.
     * </p>
     * <p>
     * This method should only be called when implementations are expected to be able to fulfil the request in a
     * performant way - if the implementation may need to lookup the underlying domain object, prefer
     * {@link #getAvatar(AvatarOwner, int)} or otherwise ensure the fetch is cheap.
     *
     * @param identifier the String version of the application-specific unique primary key for the user.
     * @param size       the size of the avatar in pixels.
     * @return the avatar at the requested size or another size if that size is not possible.
     */
    Avatar getAvatar(String identifier, int size);

    /**
     * Fetches the Avatar belonging to the given avatarOwner domain object.
     *
     * @param avatarOwner the user or other application-specific domain object to which the Avatar belongs.
     * @param size        the size of the avatar in pixels.
     * @return the Avatar.
     */
    Avatar getAvatar(AvatarOwner<T> avatarOwner, int size);

    /**
     * Optional - Returns the Avatar by its own ID.
     *
     * @throws UnsupportedOperationException for implementations that do not have AvatarIds
     */
    Avatar getAvatarById(I avatarId, int size);


    /**
     * Fetches the Avatar belonging to the given avatarOwner domain object or just applies provided fallback function.
     *
     * @param avatarOwner      the user or other application-specific domain object to which the Avatar belongs.
     * @param fallbackFunction implementation of the function that fetches avatar for the user provided by the application.
     * @param size             the size of the avatar in pixels.
     * @return the Avatar.
     */
    Avatar getAvatar(AvatarOwner<T> avatarOwner, Function<AvatarOwner<T>, Avatar> fallbackFunction, int size);

    /**
     * Bulk loading of avatars. This default implementation isn't particularly useful as it just calls the old sequential method.
     * It's up to the subclasses to implement this in a more efficient way.
     *
     * @param keyExtractor Transforms the avatar owner object into a key
     * @param avatarOwners The users for which we want to return the avatars
     * @param sizes        The desired avatar sizes
     * @param toAvatarType A transformation from the Avatar class to any desired class. Could even be just the identity function.
     * @param <K>          The type of the key used to identify the user (could be String for user key, or Long for user id, for example)
     * @param <A>          The type of the returned avatar
     */
    default <K, A> Map<K, Map<Integer, A>> getAvatarsForUsersAndSizes(
            java.util.function.Function<T, K> keyExtractor,
            Collection<AvatarOwner<T>> avatarOwners,
            List<Integer> sizes,
            java.util.function.Function<Avatar, A> toAvatarType
    ) {
        return avatarOwners.stream()
                .collect(Collectors.toMap(
                        owner -> keyExtractor.apply(owner.get()),
                        owner -> sizes.stream()
                                .collect(Collectors.toMap(size -> size, size -> toAvatarType.apply(this.getAvatar(owner, size))))));
    }

    // TODO need way for front-end to ask if/how they can edit avatar. JIRA's AvatarService method is hasCustomUserAvatar(ApplicationUser remoteUser, ApplicationUser user)
}
