package com.atlassian.plugins.avatar;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * AUI standard Avatar sizes in pixels with their t-shirt size names. Size names are designed to be used case
 * insensitively. Retina (HiDPI) sizes are typically double the edge width of the nominal size.
 */
@SuppressWarnings("UnusedDeclaration")
public enum AuiSize
{

    XSMALL(16),
    SMALL(24),
    MEDIUM(32),
    LARGE(48),
    XLARGE(64),
    XXLARGE(96),
    XXXLARGE(128);

    private static final Logger LOGGER = LoggerFactory.getLogger(AuiSize.class);

    private static final AuiSize DEFAULT = XXXLARGE;

    private final int pixelEdge;

    AuiSize(final int pixelEdge) {
        this.pixelEdge = pixelEdge;
    }

    public int getSize() {
        return pixelEdge;
    }

    /**
     * Get AvatarSize by name, case insensitive.
     *
     * @param name the t-shirt size name like xsmall
     * @return
     */
    public static AuiSize byName(String name) {
        try {
            return valueOf(name.toUpperCase());
        } catch (IllegalArgumentException e) {
            LOGGER.debug("Unknown t-shirt name requested, '{}' falling back to default size '{}'", name, DEFAULT);
            return DEFAULT;
        }
    }

    /**
     * Convenience method for returning a pixel size number for a given name, falling back to default size if the name
     * is bad.
     *
     * @param name one of the name values.
     * @return the size in pixels.
     */
    public static int toPixels(String name) {
        return byName(name).getSize();
    }
}
