var Whitelist = window.Whitelist || {};

var DOC_URL = "https://confluence.atlassian.com/display/" + (window.JIRA ? "JIRA" : "DOC") + "/Configuring+the+Whitelist";

(function ($, skate) {

    if (skate) {
        skate('whitelist-icon', {
            type: skate.types.CLASS,
            insert: function(element) {
                //after inserting to DOM we fetch favicon images if they are available, if not we use default class
                var $whitelistIcon = $(element);
                var iconUrl = $whitelistIcon.data('iconUrl');

                $whitelistIcon.removeClass(element.defaultIconClass);
                if (iconUrl) {
                    element.getImage(iconUrl);
                } else {
                    $whitelistIcon.addClass(element.defaultIconClass);
                }
            },
            prototype: {
                defaultIconClass: 'aui-icon aui-icon-small aui-iconfont-weblink',
                getImage: function(src) {
                    var $icon = $(this);
                    var img = new Image();

                    $icon.spin();

                    img.onload = function () {
                        $icon.css("background-image", "url(" + src + ")").spinStop();
                    };
                    img.onerror = function () {
                        $icon.addClass(this.defaultIconClass).spinStop();
                    }.bind(this);

                    img.src = src;
                }
            }
        });
    }


    function clearAuiMessageBar() {
        $("#aui-message-bar").empty();
    }

    function errorHandler(resp) {
        clearAuiMessageBar();
        if (resp.message === "This resource requires WebSudo.") {
            resp.message = AJS.I18n.getText("whitelist.ui.error.websudo");
        }
        AJS.messages.error({
            title: AJS.I18n.getText("whitelist.ui.error"),
            body: resp.message
        });
    }

    function createErrorResponse(errors) {
        return {
            errors: errors
        };
    }

    Whitelist.types = {
        "whitelist.application.link": {
            name: AJS.I18n.getText("whitelist.application.link"),
            test: $.noop
        },
        "whitelist.domain.name": {
            name: AJS.I18n.getText("whitelist.domain.name"),
            test: function (str) {
                return /^(?:https?:\/\/)?[a-zA-Z0-9\-\.]+(?::[0-9]+)?$/.test(str);
            }
        },
        "whitelist.exact.url": {
            name: AJS.I18n.getText("whitelist.exact.url"),
            test: function (str) {
                return /^(?:https?:\/\/)?[a-zA-Z0-9\-\.]+(?::[0-9]+)?(?:\/[\w\-\.\?~&=%]*)*$/.test(str);
            }
        },
        "whitelist.wildcard.expression": {
            name: AJS.I18n.getText("whitelist.wildcard.expression"),
            test: function (str) {
                return /^[\w\/\.\-\?:~&=%\*]*$/.test(str) && ~str.indexOf("*");
            }
        },
        "whitelist.regular.expression": {
            name: AJS.I18n.getText("whitelist.regular.expression"),
            test: function (str) {
                try {
                    new RegExp(str);
                    return true;
                } catch (e) {
                    return false;
                }
            }
        }
    };

    var selectableTypes = _.omit(Whitelist.types, "whitelist.application.link");

    Whitelist.EntryModel = AJS.RestfulTable.EntryModel.extend({
        validate: function (attrs, options) {
            options = options || {};
            if (!options.unset && attrs.allowEdit !== false) {
                if (!attrs.expression) {
                    options.status = 400;
                    return createErrorResponse({
                        expression: AJS.I18n.getText("whitelist.ui.error.noexpression")
                    });
                } else if (Whitelist.types[attrs.type].test && !Whitelist.types[attrs.type].test(attrs.expression)) {
                    options.status = 400;
                    return createErrorResponse({
                        expression: AJS.I18n.getText("whitelist.ui.error.invalidexpression")
                    });
                }
            }
        },
        save: function (attrs, options) {
            if (attrs.expression && (attrs.type === "whitelist.exact.url" || attrs.type === "whitelist.domain.name") && !/:\/\//.test(attrs.expression)) {
                attrs.expression = "http://" + attrs.expression;
            }
            if (!this._validate(attrs, _.extend({}, options, {silent: false}))) return false;
            return AJS.RestfulTable.EntryModel.prototype.save.apply(this, arguments);
        },
        changedAttributes: function (attributes) {
            var changed = {},
                current = this.toJSON();
            $.each(attributes, function (name, value) {
                if (current[name] === undefined) {
                    if (typeof value === "string") {
                        if ($.trim(value) !== "") {
                            changed[name] = value;
                        }
                    } else if ($.isArray(value)) {
                        if (value.length !== 0) {
                            changed[name] = value;
                        }
                    } else {
                        changed[name] = value;
                    }
                } else if (current[name] !== undefined && current[name] !== value) {
                    if (typeof value === "object") {
                        if (!_.isEqual(value, current[name])) {
                            changed[name] = value;
                        }
                    } else {
                        changed[name] = value;
                    }
                }
            });
            if (!_.isEmpty(changed)) {
                this.addExpand(changed);
                return changed;
            }
        }
    });

    Whitelist.ExpressionEditView = Backbone.View.extend({
        render: function (data) {
            if (this.model.get("allowEdit") !== false) {
                return Whitelist.expressionColumnEdit(data);
            } else {
                return new Whitelist.ExpressionReadView({
                    model: this.model
                }).render(data);
            }
        }
    });

    Whitelist.ExpressionReadView = Backbone.View.extend({
        render: function (data) {
            data.iconUrl = this.model.get("iconUrl");
            var $el = $(Whitelist.expressionColumnRead(data));
            return $el;
        }
    });

    Whitelist.TypeEditView = Backbone.View.extend({
        render: function (data) {
            if (this.model.get("allowEdit") !== false) {
                return aui.form.select({
                    name: data.name,
                    options: _.map(selectableTypes, function (value, key) {
                        return {
                            text: value.name,
                            value: key,
                            selected: data.value === key
                        }
                    }),
                    extraClasses: "full-width-field"
                });
            } else {
                return new Whitelist.TypeReadView({
                    model: this.model
                }).render(data);
            }
        }
    });

    Whitelist.TypeReadView = Backbone.View.extend({
        render: function (data) {
            return Whitelist.typeColumnRead({
                value: Whitelist.types[data.value].name
            });
        }
    });

    Whitelist.AllowInboundEditView = Backbone.View.extend({
        tagName: "span",
        className: "whitelist-checkbox-container",
        events: {
            "change [type=checkbox]": "updateHiddenField"
        },
        template: Whitelist.allowInboundColumnEdit,
        render: function (data) {
            data.value = !!data.value;
            this.$el.html(this.template(data));
            return this.$el;
        },
        updateHiddenField: function (e) {
            this.$("[name=allowInbound]").val(e.target.checked);
        }
    });

    Whitelist.AllowInboundReadView = Backbone.View.extend({
        render: Whitelist.allowInboundColumnRead
    });

    Whitelist.RowView = AJS.RestfulTable.Row.extend({
        initialize: function () {
            AJS.RestfulTable.Row.prototype.initialize.apply(this, arguments);
            this.allowEdit = this.allowEdit && this.model.get("allowEdit");
            this.allowDelete = this.allowDelete || this.model.get("allowDelete");
        }
    });

    Whitelist.EditRowView = AJS.RestfulTable.EditRow.extend({
        events: _.extend({
            "input": "handleInput",
            "change .select": "handleChange"
        }, AJS.RestfulTable.EditRow.prototype.events),
        handleInput: _.debounce(function (e) {
            var val = $(e.target).val();
            var select = this.$(".select");
            if (val) {
                _.every(Whitelist.types, function (type, key) {
                    if (type.test(val)) {
                        select.val(key);
                        return false;
                    }
                    return true;
                });
            }
            select.trigger("change");
        }, 500),
        handleChange: function (e) {
            var expression = this.$(".text").val();
            var error = this.$(".whitelist-error");

            if (expression) {
                error.toggleClass("hidden", !this.model.validate({
                    expression: expression,
                    type: $(e.target).val()
                }));
            } else {
                error.addClass("hidden");
            }
        },
        render: function () {
            AJS.RestfulTable.EditRow.prototype.render.apply(this, arguments);
            this.$(".select").trigger("change");
            return this;
        }
    });

    Whitelist.ToggleView = Backbone.View.extend({
        events: {
            "click": "handleClick"
        },
        initialize: function (options) {
            this.$table = options.$table;
            this.enabled = this.$el.data("enabled");
        },
        handleClick: function () {
            if (this.enabled) {
                new AJS.Dialog(400, 240)
                    .addHeader(AJS.I18n.getText("whitelist.ui.whitelist.disable.confirm.header"))
                    .addPanel(null, Whitelist.disableConfirmation())
                    .addButton(AJS.I18n.getText("whitelist.ui.confirm"), _.bind(function (dialog) {
                        this.toggleWhitelist();
                        dialog.hide();
                    }, this))
                    .addLink(AJS.I18n.getText("aui.words.cancel"), function (dialog) {
                        dialog.hide();
                    })
                    .show();
            } else {
                this.toggleWhitelist();
            }
        },
        toggleWhitelist: function () {
            this.enabled = !this.enabled;
            this.trigger(this.enabled ? "enable" : "disable");
            this.$el.text(this.enabled ? AJS.I18n.getText("whitelist.ui.whitelist.disable") : AJS.I18n.getText("whitelist.ui.whitelist.enable"));
            $.ajax({
                url: AJS.contextPath() + "/rest/whitelist/latest/" + (this.enabled ? "enable" : "disable"),
                type: "POST",
                dataType: "json",
                contentType: "application/json"
            }).done(_.bind(function () {
                clearAuiMessageBar();
                AJS.messages.success({
                    body: this.enabled ? AJS.I18n.getText("whitelist.ui.whitelist.enabled") : AJS.I18n.getText("whitelist.ui.whitelist.disabled"),
                    fadeout: true
                });
            }, this)).fail(function (xhr) {
                errorHandler(JSON.parse(xhr.responseText));
            });
        }
    });

    Whitelist.CheckerView = Backbone.View.extend({
        events: {
            "submit": "handleSubmit",
            "focus .whitelist-checker-field": "handleFocus",
            "blur .whitelist-checker-field": "handleBlur",
            "input .whitelist-checker-field": "handleInput"
        },
        initialize: function () {
            this.container = this.$el.parent();
            this.$spinner = this.$(".whitelist-checker-spinner");
            this.$inbound = this.$(".whitelist-inbound");
            this.$outbound = this.$(".whitelist-outbound");
            this.$icons = this.$spinner.add(this.$inbound).add(this.$outbound);
            this.$spinner.tipsy({
                gravity: 'se'
            });
            this.$inbound.add(this.$outbound).tipsy({
                gravity: 'w'
            });
            this.collection.on("add remove change", _.bind(this.submit, this));
        },
        handleSubmit: function (e) {
            e.preventDefault();
            this.submit();
        },
        successTemplate: Whitelist.icon({
            useIconFont: true,
            icon: "approve"
        }),
        failTemplate: Whitelist.icon({
            useIconFont: true,
            icon: "error"
        }),
        submit: _.debounce(function () {
            var params = this.$el.serializeObject();
            this.$icons.empty();
            if (params.url && params.url !== "http://") {
                if (this.$spinner.spin) {
                    this.$spinner.spin();
                }
                $.get(AJS.contextPath() + "/rest/whitelist/latest/check", params)
                    .done(_.bind(function (data) {
                        if (this.$spinner.spinStop) {
                            this.$spinner.spinStop();
                        }
                        this.$inbound.html(data.inbound ? this.successTemplate : this.failTemplate);
                        this.$outbound.html(data.outbound ? this.successTemplate : this.failTemplate);
                        if (data.inbound) {
                            this.$inbound.removeAttr("original-title");
                        } else {
                            this.$inbound.attr("original-title", AJS.I18n.getText("whitelist.ui.checker.restricted.inbound"));
                        }
                        if (data.outbound) {
                            this.$outbound.removeAttr("original-title");
                        } else {
                            this.$outbound.attr("original-title", AJS.I18n.getText("whitelist.ui.checker.restricted.outbound"));
                        }
                    }, this))
                    .fail(_.bind(function (xhr) {
                        this.$spinner.html(this.failTemplate).attr("original-title", JSON.parse(xhr.responseText).message);
                    }, this));
            }
        }, 500, true),
        handleFocus: function (e) {
            if (!e.target.value) {
                e.target.value = "http://";
            }
        },
        handleBlur: function (e) {
            if (e.target.value === "http://") {
                e.target.value = "";
            }
        },
        handleInput: _.debounce(function () {
            this.$el.submit();
        }, 500),
        disable: function () {
            this.container.addClass("hidden");
        },
        enable: function () {
            this.container.removeClass("hidden");
        }
    });

    Whitelist.RestfulTable = AJS.RestfulTable.extend({
        disable: function () {
            this.$el.addClass("hidden");
        },
        enable: function () {
            this.$el.removeClass("hidden");
        }
    });

    AJS.bindEvt(AJS.RestfulTable.Events.SERVER_ERROR, function (e, resp) {
        errorHandler(resp);
    });

    $(function () {
        $("#whitelist-doc-link").attr("href", DOC_URL);

        var restfulTable = new Whitelist.RestfulTable({
            el: "#whitelist-table",
            deleteConfirmation: Whitelist.deleteConfirmation,
            resources: {
                all: function (callback) {
                    callback($("#whitelist-data").data("whitelist"));
                },
                self: AJS.contextPath() + "/rest/whitelist/latest/"
            },
            columns: [
                {
                    id: "expression",
                    header: AJS.I18n.getText("whitelist.ui.header.expression"),
                    editView: Whitelist.ExpressionEditView,
                    readView: Whitelist.ExpressionReadView
                },
                {
                    id: "type",
                    header: Whitelist.typeHeader,
                    editView: Whitelist.TypeEditView,
                    readView: Whitelist.TypeReadView
                },
                {
                    id: "allowInbound",
                    header: Whitelist.allowInboundHeader,
                    editView: Whitelist.AllowInboundEditView,
                    readView: Whitelist.AllowInboundReadView,
                    allowEdit: false
                }
            ],
            model: Whitelist.EntryModel,
            views: {
                row: Whitelist.RowView,
                editRow: Whitelist.EditRowView
            }
        });

        var checker = new Whitelist.CheckerView({
            el: "#whitelist-checker",
            collection: restfulTable._models
        });

        new Whitelist.ToggleView({
            el: "#whitelist-enable"
        }).on("disable", _.bind(checker.disable, checker))
          .on("disable", _.bind(restfulTable.disable, restfulTable))
          .on("enable", _.bind(checker.enable, checker))
          .on("enable", _.bind(restfulTable.enable, restfulTable));
    });
})(AJS.$, window.skate);
