package com.atlassian.plugins.whitelist;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nullable;
import java.util.Collection;

/**
 * Handles all whitelist rules related data queries and modification. This includes permission checking.
 * @since 1.1
 */
@PublicApi
public interface WhitelistService
{
    /**
     * @return <code>true</code> if the whitelist is enabled, <code>false</code> otherwise
     */
    boolean isWhitelistEnabled();

    /**
     * Enable the whitelist in general. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistEnabledEvent}
     * after the modification has been finished.
     * @throws com.atlassian.plugins.whitelist.NotAuthorizedException if the current user has not the required permission
     */
    void enableWhitelist();

    /**
     * Disable the whitelist in general. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistDisabledEvent}
     * after the modification has been finished.
     * @throws com.atlassian.plugins.whitelist.NotAuthorizedException if the current user has not the required permission
     */
    void disableWhitelist();

    /**
     * Add a new whitelist rule. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistRuleAddedEvent} after
     * the modification has been finished.
     * @param whitelistRule the whitelist rule to be added
     * @return the created whitelist rule
     * @throws com.atlassian.plugins.whitelist.NotAuthorizedException if the current user has not the required permission
     */
    WhitelistRule add(WhitelistRule whitelistRule);

    /**
     * Update a given whitelist rule. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistRuleChangedEvent}
     * after the modification has been finished.
     * @param whitelistRule the updated information to be used with the whitelist rule
     * @return the updated whitelist rule
     * @throws com.atlassian.plugins.whitelist.NotAuthorizedException if the current user has not the required permission
     */
    WhitelistRule update(WhitelistRule whitelistRule);

    /**
     * Remove a whitelist rule referenced by id. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistRuleRemovedEvent}
     * after the modification has been finished.
     * @param id the id of the whitelist rule to be removed
     * @throws com.atlassian.plugins.whitelist.NotAuthorizedException if the current user has not the required permission
     */
    void remove(int id);

    /**
     * @return all known whitelist rules
     */
    Collection<WhitelistRule> getAll();

    /**
     * @param id the id of the whitelist rule
     * @return the whitelist rule with the given id or <code>null</code> if not found
     */
    @Nullable
    WhitelistRule get(int id);
}
