package com.atlassian.plugins.whitelist.migration.jira;

import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.plugins.whitelist.WhitelistManager;
import com.atlassian.plugins.whitelist.WhitelistOnOffSwitch;
import com.atlassian.plugins.whitelist.WhitelistRule;
import com.atlassian.plugins.whitelist.migration.AbstractWhitelistPluginUpgradeTask;
import com.atlassian.plugins.whitelist.LegacyWhitelistRule;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * @since 1.0
 */
public class JiraWhitelistMigration extends AbstractWhitelistPluginUpgradeTask
{
    private static final Logger logger = LoggerFactory.getLogger(JiraWhitelistMigration.class);

    private final ApplicationProperties applicationProperties;
    private final WhitelistManager whitelistManager;
    private final TransactionTemplate transactionTemplate;
    private final WhitelistOnOffSwitch whitelistOnOffSwitch;

    public JiraWhitelistMigration(final ApplicationProperties applicationProperties, final WhitelistManager whitelistManager, final TransactionTemplate transactionTemplate, final WhitelistOnOffSwitch whitelistOnOffSwitch)
    {
        this.applicationProperties = applicationProperties;
        this.whitelistManager = whitelistManager;
        this.transactionTemplate = transactionTemplate;
        this.whitelistOnOffSwitch = whitelistOnOffSwitch;
    }

    @Override
    public int getBuildNumber()
    {
        return 4;
    }

    @Override
    public String getShortDescription()
    {
        return "Migrate existing JIRA whitelist information.";
    }

    @Override
    public Collection<Message> doUpgrade() throws Exception
    {
        return transactionTemplate.execute(new TransactionCallback<Collection<Message>>()
        {
            @Override
            public Collection<Message> doInTransaction()
            {
                migrateWhitelistState();
                migrateRules();
                return Collections.emptyList();
            }

            private void migrateWhitelistState()
            {
                final boolean isDisabled = getLegacyWhitelistStateOrFalse();
                if (isDisabled)
                {
                    whitelistOnOffSwitch.disable();
                }
                else
                {
                    whitelistOnOffSwitch.enable();
                }
            }

            private void migrateRules()
            {
                final List<String> rules = readExistingData();
                final Collection<WhitelistRule> existingWhitelistRule = Collections2.transform(rules, toWhitelistRuleData());
                logger.debug("Migrating {} whitelist rules ...", existingWhitelistRule.size());
                whitelistManager.addAll(existingWhitelistRule);
            }

            private List<String> readExistingData()
            {
                final String rulesString = getLegacyWhitelistRuleOrNull();
                final String[] split = StringUtils.split(rulesString, null);
                final List<String> ret = new ArrayList<String>();
                if (split != null)
                {
                    ret.addAll(Arrays.asList(split));
                }
                return Collections.unmodifiableList(ret);
            }

            private Function<String, WhitelistRule> toWhitelistRuleData()
            {
                return new Function<String, WhitelistRule>()
                {
                    @Override
                    public WhitelistRule apply(@Nullable final String input)
                    {
                        return input != null ? new LegacyWhitelistRule(input) : null;
                    }
                };
            }

            private boolean getLegacyWhitelistStateOrFalse()
            {
                /**
                 * If the result is false, this could mean that there is no value stored at all.
                 * TODO: check if there is some data existing
                 */
                return applicationProperties.getOption("jira.whitelist.disabled");
            }

            @Nullable
            private String getLegacyWhitelistRuleOrNull()
            {
                // returns null if the propery is not existing
                return applicationProperties.getText("jira.whitelist.rules");
            }
        });
    }
}
