package com.atlassian.plugins.whitelist.migration.confluence.macros;

import com.atlassian.plugins.whitelist.LegacyWhitelistRule;
import com.atlassian.plugins.whitelist.WhitelistRule;
import com.atlassian.security.xml.libs.SecureDom4jFactory;
import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import javax.annotation.Nullable;
import java.io.StringReader;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Parse the given Bandana xml for the contained macro whitelist data.
 * @since 1.0
 */
public class BandanaMacroWhitelistXmlParser
{
    private static final boolean WHITELIST_ACTIVATED_BY_DEFAULT = true;
    private final SAXReader saxReader;

    public BandanaMacroWhitelistXmlParser()
    {
        saxReader = SecureDom4jFactory.newSaxReader();
    }

    /**
     * Parse the given xml content and extract the boolean flag whether the whitelist is enabled or not and all declared
     * whitelist rules.
     * @param xml the xml content to be parsed
     * @return the data the has been extracted from the xml content
     */
    public BandanaMacroWhitelistXmlData parseData(final String xml)
    {
        if (StringUtils.isNotEmpty(xml))
        {
            final Document document = parseDocument(checkNotNull(xml, "xml"));
            return new BandanaMacroWhitelistXmlData(isAllowAll(document), getAcceptRules(document));
        }
        else
        {
            return new BandanaMacroWhitelistXmlData(WHITELIST_ACTIVATED_BY_DEFAULT, Collections.<WhitelistRule>emptyList());
        }
    }

    private Document parseDocument(final String xml)
    {
        try
        {
            return saxReader.read(new StringReader(xml));
        }
        catch (DocumentException e)
        {
            throw new RuntimeException(e);
        }
    }

    private boolean isAllowAll(final Document document)
    {
        final Node node = document.selectSingleNode("//allowAll/text()");
        return node != null && Boolean.parseBoolean(node.getStringValue());
    }

    @SuppressWarnings("unchecked")
    private Collection<WhitelistRule> getAcceptRules(final Document document)
    {
        final List<Node> nodeList = (List<Node>) document.selectNodes("//acceptRules/string");
        final List<WhitelistRule> whitelistBeanList = Lists.transform(nodeList, transformNodeToWhitelistRule());
        return Lists.newArrayList(Iterables.filter(whitelistBeanList, Predicates.notNull()));
    }

    private Function<Node, WhitelistRule> transformNodeToWhitelistRule()
    {
        return new Function<Node, WhitelistRule>()
        {
            @Override
            public WhitelistRule apply(final @Nullable Node input)
            {
                if (input == null)
                {
                    return null;
                }

                final String textContent = input.getText();
                if (StringUtils.isEmpty(textContent))
                {
                    return null;
                }
                return new LegacyWhitelistRule(textContent);
            }
        };
    }

}
