package com.atlassian.plugins.whitelist;

import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugins.whitelist.events.ClearWhitelistCacheEvent;
import com.atlassian.plugins.whitelist.events.WhitelistDisabledEvent;
import com.atlassian.plugins.whitelist.events.WhitelistEnabledEvent;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.atlassian.util.concurrent.ResettableLazyReference;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Nullable;

/**
 * @since 1.0
 */
public class WhitelistOnOffSwitchImpl implements WhitelistOnOffSwitch, InitializingBean, DisposableBean
{
    private static final String WHITELIST_ENABLED_KEY = "com.atlassian.plugins.atlassian-whitelist-api-plugin:whitelist.enabled";
    private static final boolean ENABLED_BY_DEFAULT = true;
    private static final Logger logger = LoggerFactory.getLogger(WhitelistOnOffSwitchImpl.class);

    private final ResettableLazyReference<Boolean> enabled = new ResettableLazyReference<Boolean>()
    {
        @Override
        protected Boolean create() throws Exception
        {
            return BooleanUtils.toBooleanDefaultIfNull(loadValue(), ENABLED_BY_DEFAULT);
        }
    };
    private final PluginSettingsFactory pluginSettingsFactory;
    private final EventPublisher eventPublisher;

    public WhitelistOnOffSwitchImpl(final PluginSettingsFactory pluginSettingsFactory, final EventPublisher eventPublisher)
    {
        this.pluginSettingsFactory = pluginSettingsFactory;
        this.eventPublisher = eventPublisher;
    }

    @Override
    public void afterPropertiesSet() throws Exception
    {
        eventPublisher.register(this);
    }

    @Override
    public void destroy() throws Exception
    {
        eventPublisher.unregister(this);
    }

    @Override
    public void enable()
    {
        storeValue(true);
        enabled.reset();
        logger.debug("Whitelist has been enabled.");
        eventPublisher.publish(WhitelistEnabledEvent.INSTANCE);
    }

    @Override
    public void disable()
    {
        storeValue(false);
        enabled.reset();
        logger.debug("Whitelist has been disabled.");
        eventPublisher.publish(WhitelistDisabledEvent.INSTANCE);
    }

    @Override
    public boolean isEnabled()
    {
        return enabled.get();
    }

    @EventListener
    @SuppressWarnings("UnusedParameters")
    public void onClearWhitelistCacheEvent(final ClearWhitelistCacheEvent event)
    {
        enabled.reset();
    }

    /**
     * Return the value whereas the cases of the stored string is ignored:
     * <ul>
     *     <li><code>true</code> if the stored value is <code>true</code>, <code>on</code> or <code>yes</code></li>
     *     <li><code>false</code> if the stored value is <code>false</code>, <code>off</code> or <code>no</code></li>
     *     <li><code>null</code> otherwise</li>
     * </ul>
     * @return either <code>true</code>, <code>false</code> or <code>null</code>, depending on the stored data.
     */
    @Nullable
    private Boolean loadValue()
    {
        final Object o = settings().get(WHITELIST_ENABLED_KEY);
        return BooleanUtils.toBooleanObject(ObjectUtils.toString(o, null));
    }

    /**
     * Store the given value.
     * @param enabled the enabled state to be stored
     */
    private void storeValue(final boolean enabled)
    {
        settings().put(WHITELIST_ENABLED_KEY, Boolean.toString(enabled));
    }

    private PluginSettings settings()
    {
        return pluginSettingsFactory.createGlobalSettings();
    }
}
