package com.atlassian.plugins.cors;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.atlassian.fugue.Effect;
import com.atlassian.fugue.Option;
import com.atlassian.plugins.whitelist.InboundWhitelist;

import com.google.common.base.Predicate;

import org.apache.commons.lang.StringUtils;

/**
 * @since 1.0
 */
public class CorsFilter implements Filter
{
    private static final String ACCESS_CONTROL_ALLOW_ORIGIN = "Access-Control-Allow-Origin";
    private static final String ACCESS_CONTROL_ALLOW_CREDENTIALS = "Access-Control-Allow-Credentials";
    private static final String ACCESS_CONTROL_ALLOW_HEADERS = "Access-Control-Allow-Headers";
    private static final String ACCESS_CONTROL_ALLOW_METHODS = "Access-Control-Allow-Methods";
    private static final String HTTP_METHOD_OPTIONS = "OPTIONS";
    private static final String ORIGIN = "Origin";
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String TRUE = String.valueOf(true);

    private final InboundWhitelist inboundWhitelist;

    public CorsFilter(final InboundWhitelist inboundWhitelist)
    {
        this.inboundWhitelist = inboundWhitelist;
    }

    @Override
    public void init(FilterConfig filterConfig)
    {
    }

    @Override
    public void destroy()
    {
    }

    @Override
    public void doFilter(final ServletRequest req, final ServletResponse res, final FilterChain chain) throws IOException, ServletException
    {
        final HttpServletRequest request = (HttpServletRequest) req;
        final HttpServletResponse response = (HttpServletResponse) res;

        retrieveOrigin(request).filter(isAllowed()).foreach(addCorsResponseHeaders(response, request));

        if (!HTTP_METHOD_OPTIONS.equals(request.getMethod()))
        {
            chain.doFilter(req, res);
        }
    }

    private Option<URI> retrieveOrigin(final HttpServletRequest request)
    {
        final String origin = request.getHeader(ORIGIN);
        if (StringUtils.isBlank(origin))
        {
            return Option.none();
        }

        try
        {
            return Option.some(new URI(origin));
        }
        catch (URISyntaxException e)
        {
            return Option.none();
        }
    }

    private Predicate<URI> isAllowed()
    {
        return new Predicate<URI>()
        {
            @Override
            public boolean apply(final URI origin)
            {
                return inboundWhitelist.isAllowed(origin);
            }
        };
    }

    private Effect<URI> addCorsResponseHeaders(final HttpServletResponse response, final HttpServletRequest request)
    {
        return new Effect<URI>()
        {
            @Override
            public void apply(final URI input)
            {
                response.addHeader(ACCESS_CONTROL_ALLOW_ORIGIN, input.toString());
                response.addHeader(ACCESS_CONTROL_ALLOW_CREDENTIALS, TRUE);
                response.addHeader(ACCESS_CONTROL_ALLOW_HEADERS, CONTENT_TYPE);
                // http://www.w3.org/TR/cors/#resource-preflight-requests
                if(HTTP_METHOD_OPTIONS.equals(request.getMethod()))
                {
                    response.addHeader(ACCESS_CONTROL_ALLOW_METHODS, request.getMethod());
                }
            }
        };
    }
}
