package com.atlassian.plugin.webresource.impl.helpers;

import com.atlassian.plugin.webresource.impl.CachedCondition;
import com.atlassian.plugin.webresource.impl.RequestCache;
import com.atlassian.plugin.webresource.impl.UrlBuildingStrategy;
import com.atlassian.plugin.webresource.impl.snapshot.Bundle;
import com.google.common.base.Predicate;

import java.util.Map;

/**
 * Stateless helper functions providing basic support for resolving dependencies for resources.
 *
 * @since v3.3
 */
public class BaseHelpers {
    /**
     * Filter by web resource conditions evaluated against params.
     */
    public static Predicate<Bundle> isConditionsSatisfied(final RequestCache requestCache,
                                                          final Map<String, String> params) {
        return new Predicate<Bundle>() {
            @Override
            public boolean apply(Bundle bundle) {
                CachedCondition condition = bundle.getCondition();
                return (condition == null) || (condition.evaluateSafely(requestCache, params));
            }
        };
    }

    /**
     * Filter by web resource conditions evaluated immediately.
     */
    public static Predicate<Bundle> isConditionsSatisfied(final RequestCache requestCache, UrlBuildingStrategy urlBuilderStrategy) {
        return new Predicate<Bundle>() {
            @Override
            public boolean apply(Bundle bundle) {
                CachedCondition condition = bundle.getCondition();
                return (condition == null) || (condition.evaluateSafely(requestCache, urlBuilderStrategy));
            }
        };
    }

    /**
     * Filter that leaves only resources that have legacy conditions.
     */
    public static Predicate<Bundle> hasLegacyCondition() {
        return new Predicate<Bundle>() {
            @Override
            public boolean apply(Bundle bundle) {
                return bundle.hasLegacyConditions();
            }
        };
    }

    /**
     * Filter that leaves only resources that have legacy conditions.
     */
    public static Predicate<Bundle> hasConditions() {
        return new Predicate<Bundle>() {
            @Override
            public boolean apply(Bundle bundle) {
                return bundle.getCondition() != null;
            }
        };
    }
}
