package com.atlassian.plugin.webresource.condition;

import com.atlassian.plugin.web.Condition;
import com.atlassian.plugin.webresource.QueryParams;
import com.atlassian.plugin.webresource.impl.UrlBuildingStrategy;
import com.atlassian.plugin.webresource.url.UrlBuilder;
import com.atlassian.util.profiling.Ticker;
import com.atlassian.webresource.api.prebake.Dimensions;

import java.util.Map;

import static com.atlassian.plugin.webresource.condition.MetricsUtil.startWebConditionProfilingTimer;
import static org.apache.commons.lang3.StringUtils.EMPTY;

/**
 * Wrapper for legacy conditions
 *
 * @since v3.0
 */
class DecoratingLegacyCondition implements DecoratingCondition {
    protected final Condition legacyCondition;
    private final boolean invert;
    private final String pluginKey;
    private final String conditionClassName;

    public DecoratingLegacyCondition(Condition legacyCondition) {
        this(legacyCondition, false);
    }

    public DecoratingLegacyCondition(Condition legacyCondition, boolean invert) {
        this(legacyCondition, EMPTY, EMPTY, invert);
    }

    public DecoratingLegacyCondition(Condition legacyCondition, String pluginKey, String conditionClassName) {
        this(legacyCondition, pluginKey, conditionClassName, false);
    }

    public DecoratingLegacyCondition(Condition legacyCondition, String pluginKey, String conditionClassName, boolean invert) {
        this.legacyCondition = legacyCondition;
        this.pluginKey = pluginKey;
        this.conditionClassName = conditionClassName;
        this.invert = invert;
    }

    @Override
    public void addToUrl(UrlBuilder urlBuilder, UrlBuildingStrategy urlBuilderStrategy) {
    }

    @Override
    public Dimensions computeDimensions() {
        return Dimensions.empty();
    }

    @Override
    public boolean shouldDisplay(QueryParams params) {
        // If this is being called for a legacy condition, it means that the shouldDisplayImmediate() returned true
        // and the resource was included on the served page - so return true.
        return true;
    }

    @Override
    public boolean shouldDisplayImmediate(Map<String, Object> params, UrlBuildingStrategy urlBuilderStrategy) {
        try (Ticker ignored = startWebConditionProfilingTimer(pluginKey, conditionClassName)) {
            boolean shouldDisplay = this.legacyCondition.shouldDisplay(params);
            return this.invert ? !shouldDisplay : shouldDisplay;
        }
    }

    @Override
    public boolean canEncodeStateIntoUrl() {
        return false;
    }

    @Override
    public DecoratingCondition invertCondition() {
        return new DecoratingLegacyCondition(legacyCondition, pluginKey, conditionClassName, !invert);
    }
}
