package com.atlassian.plugin.webresource.transformer;

import com.atlassian.plugin.elements.ResourceLocation;
import com.atlassian.plugin.webresource.QueryParams;
import com.atlassian.plugin.webresource.impl.UrlBuildingStrategy;
import com.atlassian.plugin.webresource.impl.helpers.ResourceServingHelpers;
import com.atlassian.plugin.webresource.impl.support.Content;
import com.atlassian.plugin.webresource.url.UrlBuilder;
import com.atlassian.webresource.api.prebake.Coordinate;
import com.atlassian.webresource.api.prebake.DimensionAwareWebResourceTransformerFactory;
import com.atlassian.webresource.api.prebake.Dimensions;

/**
 * {@link com.atlassian.plugin.webresource.transformer.StaticTransformers} implementation.
 *
 * @since v3.1.0
 */
public class DefaultStaticTransformers implements StaticTransformers
{
    private final StaticTransformersSupplier staticTransformersSupplier;

    public DefaultStaticTransformers(StaticTransformersSupplier staticTransformersSupplier)
    {
        this.staticTransformersSupplier = staticTransformersSupplier;
    }

    @Override
    public Dimensions computeDimensions() {
        return staticTransformersSupplier.computeDimensions();
    }

    @Override
    public void addToUrl(String locationType, TransformerParameters transformerParameters, UrlBuilder urlBuilder, UrlBuildingStrategy urlBuildingStrategy)
    {
        for (DimensionAwareWebResourceTransformerFactory transformerFactory : transformersForType(locationType))
        {
            urlBuildingStrategy.addToUrl(transformerFactory.makeUrlBuilder(transformerParameters), urlBuilder);
        }
    }

    @Override
    public Content transform(Content content, TransformerParameters transformerParameters, ResourceLocation resourceLocation, String filePath,
                             QueryParams queryParams, String sourceUrl) {
        for (DimensionAwareWebResourceTransformerFactory transformerFactory : transformersForLocation(resourceLocation))
        {
            TransformableResource tr = new TransformableResource(
                    resourceLocation,
                    filePath,
                    ResourceServingHelpers.asDownloadableResource(content)
            );
            content = ResourceServingHelpers.asContent(transformerFactory.makeResourceTransformer(transformerParameters)
                    .transform(tr, queryParams), null, true);
        }
        return content;
    }

    private Iterable<DimensionAwareWebResourceTransformerFactory> transformersForType(final String locationType)
    {
        return staticTransformersSupplier.get(locationType);
    }

    private Iterable<DimensionAwareWebResourceTransformerFactory> transformersForLocation(final ResourceLocation location)
    {
        return staticTransformersSupplier.get(location);
    }
}
