package com.atlassian.plugin.webresource.impl;

import com.atlassian.json.marshal.Jsonable;
import com.atlassian.plugin.webresource.bigpipe.BigPipe;
import com.atlassian.plugin.webresource.impl.snapshot.Snapshot;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Set;

import static org.apache.commons.lang.StringUtils.join;

/**
 * State maintained during single request.
 *
 * @since 3.3
 */
public class RequestState
{
    protected final LinkedHashSet<String> included;
    protected final LinkedHashSet<String> excluded;
    protected final LinkedHashMap<String, Jsonable> includedData;
    protected final Set<String> excludedData;
    protected final RequestCache cache;
    private final UrlBuildingStrategy urlBuildingStrategy;
    /** this is the ms-since-epoc that the dealine expires */
    private volatile long bigPipDeadline;
    /** request-local */
    private final BigPipe pipe = new BigPipe();

    protected final Globals globals;

    private RequestState(Globals globals, LinkedHashSet<String> included, LinkedHashSet<String> excluded,
        LinkedHashMap<String, Jsonable> includedData, Set<String> excludedData, RequestCache cache, UrlBuildingStrategy urlBuildingStrategy)
    {
        this.globals = globals;
        this.included = included;
        this.excluded = excluded;
        this.includedData = includedData;
        this.excludedData = excludedData;
        this.cache = cache;
        this.urlBuildingStrategy = urlBuildingStrategy;
        this.bigPipDeadline = System.currentTimeMillis() + globals.getConfig().getDefaultBigPipeDeadline().toMillis();
    }

    public RequestState(Globals globals, UrlBuildingStrategy urlBuildingStrategy)
    {
        this(globals, new LinkedHashSet<>(), new LinkedHashSet<>(), new LinkedHashMap<>(),
                new HashSet<>(), new RequestCache(globals), urlBuildingStrategy);
    }

    public UrlBuildingStrategy getUrlStrategy()
    {
        return urlBuildingStrategy;
    }

    public RequestState deepClone()
    {
        RequestState clone = new RequestState(globals, new LinkedHashSet<>(included), new LinkedHashSet<>(excluded),
                new LinkedHashMap<>(includedData), new HashSet<>(excludedData), cache, urlBuildingStrategy);
        clone.setBigPipeDeadline(bigPipDeadline);
        return clone;
    }

    public void setBigPipeDeadline(long deadline) {
        this.bigPipDeadline = deadline;
    }
    public long getBigPipeDeadline()
    {
        return bigPipDeadline;
    }

    public LinkedHashSet<String> getIncluded()
    {
        return included;
    }

    public LinkedHashMap<String, Jsonable> getIncludedData()
    {
        return includedData;
    }

    public Set<String> getExcludedData()
    {
        return excludedData;
    }

    public LinkedHashSet<String> getExcluded()
    {
        return excluded;
    }

    public BigPipe getBigPipe()
    {
        return pipe;
    }

    /**
     * Called after generating urls, to clear current state and remember already included resources.
     */
    public void clearIncludedAndUpdateExcluded(LinkedHashSet<String> excludedResolved)
    {
        included.clear();
        excluded.addAll(excludedResolved);

        excludedData.addAll(includedData.keySet());
        includedData.clear();
    }

    public Globals getGlobals()
    {
        return globals;
    }

    /**
     * Get all bundles.
     * <p>
     * It is another layer of cache over the `globals.getBundles()` because it is used very heavily, to avoid any
     * performance drawbacks of atomic reference in the `globals.getBundles()`.
     */
    public Snapshot getSnapshot()
    {
        return getCache().getSnapshot();
    }

    public RequestCache getCache()
    {
        return cache;
    }

    public String toString()
    {
        return "[" + join(included, ", ") + "] - [" + join(excluded, ", ") + "]";
    }
}