package com.atlassian.plugin.webresource.url;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Immutable container for url parameters
 *
 * @since v3.0.5
 */
public class UrlParameters
{
    private static final UrlParameters EMPTY_CDN_OK = new UrlParameters(true);
    private static final UrlParameters EMPTY_CDN_TAINTED = new UrlParameters(false);

    private final List<String> hashes;
    private final Map<String, String> queryString;
    private final boolean isCdnSupported;

    public static UrlParameters of(boolean isCdnSupported)
    {
        return isCdnSupported ? EMPTY_CDN_OK : EMPTY_CDN_TAINTED;
    }

    public static UrlParameters of(boolean isCdnSupported, List<String> hashes, Map<String, String> queryString)
    {
        return new UrlParameters(isCdnSupported, hashes, queryString);
    }

    private UrlParameters(boolean isCdnSupported)
    {
        this(isCdnSupported, Collections.<String>emptyList(), Collections.<String, String>emptyMap());
    }

    private UrlParameters(boolean isCdnSupported, List<String> hashes, Map<String, String> queryString)
    {
        this.isCdnSupported = isCdnSupported;
        this.hashes = ImmutableList.copyOf(hashes);
        this.queryString = ImmutableMap.copyOf(queryString);
    }

    public List<String> allHashes()
    {
        return hashes;
    }

    public Map<String, String> toQueryString()
    {
        return queryString;
    }

    public boolean isCdnSupported()
    {
        return isCdnSupported;
    }

    public UrlParameters merge(UrlParameters other)
    {
        Map<String, String> mergedQueryString = Maps.newHashMap(queryString);
        mergedQueryString.putAll(other.queryString);
        List<String> mergedHashes = ImmutableList.copyOf(Iterables.concat(hashes, other.hashes));
        return new UrlParameters(isCdnSupported && other.isCdnSupported(), mergedHashes, mergedQueryString);
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        UrlParameters that = (UrlParameters) o;

        if (isCdnSupported != that.isCdnSupported) return false;
        if (!hashes.equals(that.hashes)) return false;
        if (!queryString.equals(that.queryString)) return false;

        return true;
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(hashes.hashCode(), queryString.hashCode(), Boolean.valueOf(isCdnSupported).hashCode());
    }
}
