package com.atlassian.plugin.webresource.url;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.util.List;
import java.util.Map;

/**
 * Implementation of UrlBuilder
 *
 * @since v3.0
 */
public class DefaultUrlBuilder implements UrlBuilder
{
    private final List<String> hashes;
    private final Map<String, String> queryString;
    private boolean isCdnSupported = true;
    private static final Interner<String> hashInterner = Interners.newWeakInterner();

    public DefaultUrlBuilder()
    {
        this.hashes = Lists.newLinkedList();
        this.queryString = Maps.newLinkedHashMap();
    }

    @Override
    public void addToHash(String name, Object value)
    {
        hashes.add(hashInterner.intern(String.valueOf(value)));
    }

    @Override
    public void addToQueryString(String key, String value)
    {
        queryString.put(key, value);
    }

    /** Mark this resource as not deliverable via CDN */
    public void taint()
    {
        isCdnSupported = false;
    }

    public UrlParameters build()
    {
        return UrlParameters.of(isCdnSupported, hashes, queryString);
    }

    /**
     * Add its state to another UrlBuilder.
     */
    public void addTo(DefaultUrlBuilder urlBuilder)
    {
        for (Map.Entry<String, String> entry : queryString.entrySet())
        {
            urlBuilder.addToQueryString(entry.getKey(), entry.getValue());
        }
        for (String hash : hashes)
        {
            urlBuilder.addToHash(null, hash);
        }
        if (!isCdnSupported)
        {
            urlBuilder.taint();
        }
    }
}
