package com.atlassian.plugin.webresource.http;

import com.atlassian.annotations.Internal;
import com.atlassian.plugin.cache.filecache.Cache;
import com.atlassian.plugin.servlet.util.LastModifiedHandler;
import com.atlassian.plugin.webresource.Content;
import com.atlassian.plugin.webresource.Globals;
import com.atlassian.plugin.webresource.RequestCache;
import com.atlassian.plugin.webresource.Resource;
import com.atlassian.plugin.webresource.support.NullOutputStream;
import com.atlassian.plugin.webresource.support.http.BaseController;
import com.atlassian.plugin.webresource.support.http.BaseRouter;
import com.atlassian.plugin.webresource.support.http.Request;
import com.atlassian.plugin.webresource.support.http.Response;
import com.atlassian.sourcemap.SourceMap;
import com.google.common.base.Supplier;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Date;

import static com.atlassian.plugin.webresource.Helpers.buildEmptyContent;
import static com.atlassian.plugin.webresource.Helpers.getResource;
import static com.atlassian.plugin.webresource.Helpers.getResources;
import static com.atlassian.plugin.webresource.Helpers.isConditionSatisfied;
import static com.atlassian.plugin.webresource.Helpers.resolveBundles;
import static com.atlassian.plugin.webresource.Helpers.selectForBatch;
import static com.atlassian.plugin.webresource.Helpers.transform;
import static com.atlassian.sourcemap.Util.generateSourceMapComment;

/**
 * WARNING Do not use it, it will be removed in the next version!
 *
 * Handles HTTP requests.
 *
 * @since 3.3
 */
@Internal
public class Controller extends BaseController
{
    private final RequestCache requestCache;

    public Controller(Globals globals, Request request, Response response)
    {
        super(globals, request, response);
        this.requestCache = new RequestCache();
    }

    /**
     * Serves single Resource.
     */
    public void serveResource(String completeKey, String resourceName)
    {
        serveResource(getResource(globals, requestCache, completeKey, resourceName), true);
    }

    /**
     * Serves Source Map for single Resource.
     */
    public void serveResourceSourceMap(String completeKey, String resourceName)
    {
        serveSourceMap(getResource(globals, requestCache, completeKey, resourceName));
    }

    /**
     * Serves Batch (all types of Batch - Web Resource Batch, Super Batch, Context Batch).
     *
     * @param included keys of included Web Resources.
     * @param excluded keys of excluded Web Resources.
     * @param type type of Batch.
     * @param resolveDependencies if dependencies should be resolved.
     * @param isCachingEnabled if it should be cached.
     */
    public void serveBatch(final Collection<String> included, final Collection<String> excluded, final String type
        , final boolean resolveDependencies, final boolean includeLegacy, boolean isCachingEnabled)
    {
        serveResources(new Supplier<Collection<Resource>>()
        {
            @Override
            public Collection<Resource> get()
            {
                Collection<String> bundles = resolveBundles(globals, included, excluded, resolveDependencies
                    , includeLegacy, request.getParams());
                return selectForBatch(getResources(globals, requestCache, bundles), type, request.getParams());
            }
        }, isCachingEnabled);
    }

    /**
     * Serves Source Map for Batch.
     */
    public void serveBatchSourceMap(final Collection<String> included, final Collection<String> excluded
        , final String type, final boolean resolveDependencies, final boolean includeLegacy)
    {
        serveResourcesSourceMap(new Supplier<Collection<Resource>>()
        {
            @Override
            public Collection<Resource> get()
            {
                Collection<String> bundles = resolveBundles(globals, included, excluded, resolveDependencies
                    , includeLegacy, request.getParams());
                return selectForBatch(getResources(globals, requestCache, bundles), type, request.getParams());
            }
        });
    }

    /**
     * Serves Resource relative to Batch.
     */
    public void serveResourceRelativeToBatch(Collection<String> included, Collection<String> excluded,
            String resourceName, boolean resolveDependencies, boolean includeLegacy)
    {
        Collection<String> bundles = resolveBundles(globals, included, excluded, resolveDependencies, includeLegacy,
                request.getParams());
        serveResource(getResource(globals, requestCache, bundles, resourceName), true);
    }

    /**
     * Serves Source Map for Resource relative to Batch.
     */
    public void serveResourceRelativeToBatchSourceMap(Collection<String> included, Collection<String> excluded,
            String resourceName, boolean resolveDependencies, boolean includeLegacy)
    {
        Collection<String> bundles = resolveBundles(globals, included, excluded, resolveDependencies, includeLegacy,
                request.getParams());
        serveSourceMap(getResource(globals, requestCache, bundles, resourceName));
    }

    /**
     * Serves Source Code of the Resource.
     */
    public void serveSource(String completeKey, String resourceName)
    {
        serveResource(getResource(globals, requestCache, completeKey, resourceName), false);
    }

    /*
     * Helpers.
     */

    /**
     * Serves Resource.
     *
     * @param applyTransformations if transformations should be applied.
     */
    protected void serveResource(Resource resource, boolean applyTransformations)
    {
        if (handleNotFoundRedirectAndNotModified(resource))
        {
            return;
        }
        Content content;
        if (isConditionSatisfied(resource.getParent(), request.getParams()))
        {
            if (applyTransformations)
            {
                content = transform(globals, resource, request.getParams());
            }
            else
            {
                content = resource.getContent();
            }
        }
        else
        {
            content = buildEmptyContent(resource.getContent());
        }
        sendCached(request.getUrl(), content, false);
    }

    /**
     * Serves resources.
     */
    protected void serveResources(Supplier<Collection<Resource>> resources, boolean isCachingEnabled)
    {
        String type = request.getType();
        Content content = transform(globals, type, resources, request.getParams());
        sendCached(request.getUrl(), content, isCachingEnabled);
    }

    /**
     * Serves Source Map.
     */
    protected void serveSourceMap(Resource resource)
    {
        String resourcePath = globals.getRouter().resourcePathFromSourceMapPath(request.getPath());
        String cacheKey = BaseRouter.buildUrl(resourcePath, request.getParams());
        if (handleNotFoundRedirectAndNotModified(resource))
        {
            return;
        }
        Content content = transform(globals, resource, request.getParams());
        sendCached(cacheKey, content);
    }

    /**
     * Serves Source Map for Batch of Resources.
     */
    private void serveResourcesSourceMap(Supplier<Collection<Resource>> resources)
    {
        String resourcePath = globals.getRouter().resourcePathFromSourceMapPath(request.getPath());
        String cacheKey = BaseRouter.buildUrl(resourcePath, request.getParams());
        String type = Request.getType(resourcePath);
        Content content = transform(globals, type, resources, request.getParams());
        sendCached(cacheKey, content);
    }

    /**
     * Handle not found resources, redirects and not modified resources.
     * @return true if request has been handled and no further processing needed.
     */
    protected boolean handleNotFoundRedirectAndNotModified(Resource resource)
    {
        if (resource == null)
        {
            response.sendError(404);
            return true;
        }
        if (checkIfCachedAndNotModified(resource.getParent().getUpdatedAt()))
        {
            return true;
        }
        if (resource.isRedirect())
        {
            response.sendRedirect(resource.getLocation(), resource.getContentType());
            return true;
        }
        return false;
    }

    /**
     * Check if resource is not modified and replies with not-modified response if so.
     *
     * @param updatedAt when resource has been updated.
     * @return true if Resources is not modified and no further processing is needed.
     */
    protected boolean checkIfCachedAndNotModified(Date updatedAt)
    {
        LastModifiedHandler lastModifiedHandler = new LastModifiedHandler(updatedAt);
        return request.isCacheable() && lastModifiedHandler.checkRequest(request.getOriginalRequest(),
                response.getOriginalResponse());
    }

    /**
     * Serve cached Content.
     */
    protected void sendCached(String cacheKey, Content content)
    {
        sendCached(cacheKey, content, true);
    }

    /**
     * Cached content.
     */
    protected interface CachedContent
    {
        public void writeTo(OutputStream contentOut, OutputStream sourceMapOut, final String contentType,
                    final boolean isSourceMapEnabled, final String sourceMapUrl);
    }

    /**
     * Adapter to turn Content with Source Map as Object into Source Map as Stream and cache it.
     */
    protected CachedContent buildCachedContent(final String cacheKey, final Content content, final boolean isCacheEnabled)
    {
        class CachedContentImpl implements CachedContent
        {
            public void writeTo(OutputStream contentOut, OutputStream sourceMapOut, final String contentType,
                    final boolean isSourceMapEnabled, final String sourceMapUrl)
            {
                cache(cacheKey, contentOut, sourceMapOut, new Cache.TwoStreamProvider()
                {
                    @Override
                    public void write(OutputStream contentOut, OutputStream sourceMapOut)
                    {
                        // The contentOut or sourceMapOut could be null, this is done to optimize streaming and not
                        // serve stuff that's
                        // not needed.
                        if (contentOut == null)
                        {
                            contentOut = new NullOutputStream();
                        }
                        try
                        {
                            SourceMap sourceMap = content.writeTo(contentOut, isSourceMapEnabled);
                            if (isSourceMapEnabled && sourceMap != null)
                            {
                                contentOut.write(("\n" + generateSourceMapComment(sourceMapUrl,
                                        contentType)).getBytes());
                                if (sourceMapOut != null)
                                {
                                    sourceMapOut.write(sourceMap.generate().getBytes());
                                }
                            }
                        }
                        catch (IOException e)
                        {
                            throw new RuntimeException(e);
                        }
                    }
                }, isSourceMapEnabled, isCacheEnabled);
            }

            private void cache(String key, OutputStream contentOut, OutputStream sourceMapOut,
                    final Cache.TwoStreamProvider twoStreamProvider, boolean isSourceMapEnabled, boolean isCacheEnabled)
            {
                if (request.isCacheable() && isCacheEnabled)
                {
                    if (isSourceMapEnabled)
                    {
                        globals.getCache().cacheTwo("http", key, contentOut, sourceMapOut, twoStreamProvider);
                    }
                    else
                    {
                        // Switching to single stream cache if there's no source map, to improve performance.
                        globals.getCache().cache("http", key, contentOut, new Cache.StreamProvider()
                        {
                            @Override
                            public void write(OutputStream out)
                            {
                                twoStreamProvider.write(out, null);
                            }
                        });
                    }
                }
                else
                {
                    twoStreamProvider.write(contentOut, sourceMapOut);
                }
            }
        }
        return new CachedContentImpl();
    }

    // The last attribute `isCacheEnabled` would be removed in the next version, currently it's needed to forcefully
    // disabling cache for some resources.
    @Deprecated
    protected void sendCached(final String cacheKey, final Content content, final boolean isCacheEnabled)
    {
        CachedContent cachedContent = buildCachedContent(cacheKey, content, isCacheEnabled);
        if ("map".equals(request.getType()))
        {
            // Serving Source Maps.
            response.setContentTypeIfNotBlank(request.getContentType());

            String resourcePath = globals.getRouter().resourcePathFromSourceMapPath(request.getPath());
            String resourceContentType = content.getContentType() != null ? content.getContentType() : globals
                    .getConfig().getContentType(resourcePath);

            String sourceMapUrl = request.getUrl();

            cachedContent.writeTo(null, response.getOutputStream(), resourceContentType, isSourceMapEnabled(),
                    sourceMapUrl);
        }
        else
        {
            // Serving Resources.
            String contentType = content.getContentType() != null ? content.getContentType() : request.getContentType();
            response.setContentTypeIfNotBlank(contentType);

            String sourceMapUrl = globals.getRouter().sourceMapUrl(request.getPath(), request.getParams());

            cachedContent.writeTo(response.getOutputStream(), null, contentType, isSourceMapEnabled(), sourceMapUrl);
        }
    }

    /**
     * If Source Map is enabled for the current Request.
     */
    protected boolean isSourceMapEnabled()
    {
        // The `isSourceMapEnabled` used to choose single stream or double stream cache,
        // and as soon the cache should be chosen
        // before of the resolution of the resource, it can't use any data from resource (so it can't use `resource
        // .getType()`).
        // The decision should be based purely on data available in the request.
        String resourcePath = "map".equals(request.getType()) ?
                globals.getRouter().resourcePathFromSourceMapPath(request.getPath()) : request.getPath();
        String type = Request.getType(resourcePath);
        return globals.getConfig().isSourceMapEnabledFor(type);
    }
}
