package com.atlassian.plugin.webresource;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.webresource.url.UrlParameters;
import com.atlassian.util.concurrent.LazyReference;

import java.util.Collections;
import java.util.Map;

/**
 * Represents a single plugin resource.
 *
 * It provides methods to parse and generate urls to locate a single plugin resource.
 *
 * Note: This PluginResource does not use it's parameters in generating the url. 
 *
 * @since 2.2
 */
public class SinglePluginResource extends AbstractPluginResource
{
    private final String resourceName;
    private final String moduleCompleteKey;
    private final boolean cached;
    private final LazyReference<String> type;
    
    public SinglePluginResource(Globals globals, final String resourceName, final String moduleCompleteKey, final boolean cached, final boolean isCdnSupported)
    {
        this(globals, resourceName, moduleCompleteKey, cached, UrlParameters.of(isCdnSupported), Collections.<String, String>emptyMap());
    }

    public SinglePluginResource(Globals globals, final String resourceName, final String moduleCompleteKey, final boolean cached, final UrlParameters urlParams, final Map<String, String> params)
    {
        super(globals, urlParams, params, Collections.<BatchedWebResourceDescriptor>emptySet());
        this.resourceName = resourceName;
        this.moduleCompleteKey = moduleCompleteKey;
        this.cached = cached;
        this.type = new LazyReference<String>() {
            @Override
            protected String create() throws Exception
            {
                return ResourceUtils.getType(resourceName);
            }
        };
    }

    @Override
    public String getResourceName()
    {
        return resourceName;
    }

    @Override
    public String getCompleteKey()
    {
        return moduleCompleteKey;
    }

    @Override
    public String getVersion()
    {
        final Plugin plugin = globals.getConfig().getIntegration().getPluginAccessor().getEnabledPluginModule(getCompleteKey()).getPlugin();
        return plugin.getPluginInformation().getVersion();
    }

    @Override
    public String getUrl(boolean isAbsolute)
    {
        return globals.getRouter().cloneWithNewUrlMode(isAbsolute).urlFor(this);
    }

    @Override
    public boolean isCacheSupported()
    {
        return cached;
    }

    @Override
    public String getType()
    {
        return type.get();
    }

    @Override
    public Map<String, String> getParamsWithBatchableParams()
    {
        return urlParams.toQueryString();
    }
}
